// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.utils;

import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.xml.XmlFile;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.contexts.model.LocalAnnotationModel;
import com.intellij.spring.contexts.model.LocalModel;
import com.intellij.spring.contexts.model.graph.LocalModelDependency;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.jam.javaConfig.ContextJavaBean;
import com.intellij.spring.model.jam.stereotype.SpringPropertySource;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.spring.model.jam.utils.filters.SpringContextFilter;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.PairProcessor;
import com.intellij.util.Processor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * @author Yann C&eacute;bron
 */
public abstract class SpringJamUtils {

  public static SpringJamUtils getInstance() {
    return ApplicationManager.getApplication().getService(SpringJamUtils.class);
  }

  /**
   * @Bean-annotated beans.
   */
  @NotNull
  public abstract List<ContextJavaBean> getContextBeans(@NotNull PsiClass beanClass,
                                                        @Nullable Set<String> activeProfiles);
  /**
   * XML config files defined with @ImportResource annotations.
   */
  @NotNull
  public abstract Set<XmlFile> getImportedResources(@NotNull PsiClass psiClass, Module... contexts);

  /**
   * Processes {@code @ImportResource} annotation. For each element of annotation value XML config files defined by this element and
   * PsiElement defining this element are passed to processor.
   *
   * @param psiClass  Class with {@code @ImportResource} annotation to visit.
   * @param processor Processor, both pair elements are guaranteed to be non-{@code null}.
   */
  public abstract boolean processImportedResources(@NotNull PsiClass psiClass,
                                                   @NotNull Processor<Pair<List<XmlFile>, ? extends PsiElement>> processor,
                                                   Module... contexts);

  /**
   * Classes (@Configuration) defined with @Import annotations.
   */
  @NotNull
  public abstract Set<PsiClass> getImportedClasses(@NotNull PsiClass clazz,
                                                   @Nullable Module module);

  /**
   * Processes {@code @Import} annotation. For each element of annotation value Java config files defined by this element and
   * PsiElement defining this element are passed to processor.
   *
   * @param clazz     Class with {@code @Import} annotation to visit.
   * @param processor Processor, both pair elements are guaranteed to be non-{@code null}.
   */
  public abstract boolean processImportedClasses(@NotNull PsiClass clazz,
                                                 @NotNull Processor<Pair<PsiClass, ? extends PsiElement>> processor);

  @NotNull
  public abstract List<? extends SpringBeansPackagesScan> getBeansPackagesScan(@NotNull PsiClass psiClass);

  /**
   * @PropertySource, @PropertySources()
   */
  @NotNull
  public abstract Collection<SpringPropertySource> getPropertySources(@NotNull PsiClass psiClass);

  /**
   * Search configuration beans (<component-scan/>, @ComponentScan, @Repositories, etc.)
   * which loaded beans (beansInModel parameter) in model.
   */
  @NotNull
  public abstract Set<CommonModelElement> findStereotypeConfigurationBeans(@NotNull CommonSpringModel model,
                                                                           @NotNull List<? extends SpringBeanPointer> beansInModel,
                                                                           @Nullable Module module);

  /**
   * Filter components loaded by componentScan (analyses packages, include/exclude filters).
   */
  @NotNull
  public abstract Set<CommonSpringBean> filterComponentScannedStereotypes(@NotNull Module module,
                                                                          @NotNull SpringBeansPackagesScan componentScan,
                                                                          @NotNull List<? extends CommonSpringBean> allComponents);


  public abstract Set<CommonSpringBean> filterComponentScannedStereotypes(@NotNull Module module,
                                                                          @NotNull List<? extends CommonSpringBean> allComponents,
                                                                          @NotNull Set<PsiPackage> psiPackages,
                                                                          boolean useDefaultFilters,
                                                                          @NotNull Set<SpringContextFilter.Exclude> excludeContextFilters,
                                                                          @NotNull Set<SpringContextFilter.Include> includeContextFilters);

  @Nullable
  public abstract SpringStereotypeElement findStereotypeElement(@NotNull PsiClass psiClass);

  /**
   * Processes all resolved {@code @Enable...} annotations.
   *
   * @param psiClass  Class with @Enable... annotations to visit.
   * @param processor Processor, both pair elements are guaranteed to be non-{@code null}.
   */
  public abstract boolean processCustomAnnotations(@NotNull PsiClass psiClass,
                                                   @NotNull Processor<Pair<PsiClass, LocalModelDependency>> processor);

  /**
   * Allows extending {@link LocalAnnotationModel} with customized dependent models, e.g. {@code @EnableXYZ}-style models.
   *
   * @param localAnnotationModel Local annotation model to visit custom dependent local models for.
   * @param processor            Processor instance.
   * @return Processor result.
   */
  public abstract boolean processCustomDependentLocalModels(LocalAnnotationModel localAnnotationModel,
                                                            PairProcessor<? super LocalModel, ? super LocalModelDependency> processor) ;
}
