/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts.dirtiesContexts;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamEnumAttributeElement;
import com.intellij.jam.JamService;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.reflect.*;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public abstract class SpringTestingDirtiesContext implements JamElement {
  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = JamService.ANNO_META_KEY.subKey("SpringTestingDirtiesContext");
  public static final SemKey<SpringTestingDirtiesContext> JAM_KEY = JamService.JAM_ELEMENT_KEY.subKey("SpringTestingDirtiesContext");

  public static final JamClassMeta<SpringTestingDirtiesContext> CLASS_META =
    new JamClassMeta<>(null, SpringTestingDirtiesContext.class, JAM_KEY);
  public static final JamMethodMeta<SpringTestingDirtiesContext> METHOD_META =
    new JamMethodMeta<>(null, SpringTestingDirtiesContext.class, JAM_KEY);

  public static final JamAttributeMeta<JamEnumAttributeElement<MethodMode>> METHOD_MODE_ATTR_META =
    JamAttributeMeta.singleEnum("methodMode", MethodMode.class);
  public static final JamAttributeMeta<JamEnumAttributeElement<ClassMode>> CLASS_MODE_ATTR_META =
    JamAttributeMeta.singleEnum("classMode", ClassMode.class);
  public static final JamAttributeMeta<JamEnumAttributeElement<HierarchyMode>> HIERARCHY_MODE_ATTR_META =
    JamAttributeMeta.singleEnum("hierarchyMode", HierarchyMode.class);

  public static final JamAnnotationArchetype ARCHETYPE = new JamAnnotationArchetype()
    .addAttribute(METHOD_MODE_ATTR_META)
    .addAttribute(CLASS_MODE_ATTR_META)
    .addAttribute(HIERARCHY_MODE_ATTR_META);

  public static final JamAnnotationMeta ANNO_META =
    new JamAnnotationMeta(SpringAnnotationsConstants.DIRTIES_CONTEXT, ARCHETYPE, JAM_ANNO_META_KEY);

  static {
    CLASS_META.addAnnotation(ANNO_META);
    METHOD_META.addAnnotation(ANNO_META);
  }

  private final PsiMember myPsiMember;
  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;


  @SuppressWarnings("unused")
  public SpringTestingDirtiesContext(@NotNull PsiMember psiMember) {
    myPsiMember = psiMember;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiMember);
  }

  @SuppressWarnings("unused")
  public SpringTestingDirtiesContext(PsiAnnotation annotation) {
    myPsiMember = PsiTreeUtil.getParentOfType(annotation, PsiClass.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  public ClassMode getClassMode() {
    final ClassMode classMode = getClassModeElement().getValue();
    if (classMode != null) {
      return classMode;
    }
    return ClassMode.AFTER_CLASS;
  }

  public MethodMode getMethodMode() {
    final MethodMode methodMode = getMethodModeElement().getValue();
    if (methodMode != null) {
      return methodMode;
    }
    return MethodMode.AFTER_METHOD;
  }

  public HierarchyMode getHierarchyMode() {
    final HierarchyMode methodMode = getHierarchyModeElement().getValue();
    if (methodMode != null) {
      return methodMode;
    }
    return HierarchyMode.EXHAUSTIVE;
  }

  public JamEnumAttributeElement<ClassMode> getClassModeElement() {
    return CLASS_MODE_ATTR_META.getJam(myPsiAnnotation) ;
  }

  public JamEnumAttributeElement<MethodMode> getMethodModeElement() {
    return METHOD_MODE_ATTR_META.getJam(myPsiAnnotation);
  }

  public JamEnumAttributeElement<HierarchyMode> getHierarchyModeElement() {
    return HIERARCHY_MODE_ATTR_META.getJam(myPsiAnnotation);
  }

  @Nullable
  public PsiMember getPsiElement() {
    return myPsiMember;
  }

  @JamPsiValidity
  public abstract boolean isPsiValid();

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }
}
