// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.JamPomTarget;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.jam.reflect.JamStringAttributeMeta;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.pom.references.PomService;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiModifierListOwner;
import com.intellij.psi.PsiTarget;
import com.intellij.psi.targets.AliasingPsiTarget;
import com.intellij.psi.targets.AliasingPsiTargetUtil;
import com.intellij.spring.model.jam.JamPsiClassSpringBean;
import com.intellij.spring.model.jam.javaConfig.ContextJavaBean;
import com.intellij.spring.model.jam.javaConfig.SpringJavaBean;
import com.intellij.util.containers.SmartHashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.intellij.jam.JamService.CHECK_METHOD;

public abstract class SpringStereotypeElement extends JamPsiClassSpringBean {
  private static final Map<String, JamAnnotationMeta> annotationMetaMap = new HashMap<>();
  private JamAnnotationMeta myMeta;

  private static final JamStringAttributeMeta.Single<String> NAME_VALUE_META = JamAttributeMeta.singleString("value");

  public List<? extends SpringJavaBean> getBeans() {
    final List<SpringJavaBean> beans = new ArrayList<>();
    final JamService jamService = JamService.getJamService(getPsiManager().getProject());

    JamCommonUtil.processSuperClassList(getPsiElement(), new SmartHashSet<>(), psiClass -> {
      beans.addAll(jamService.getAnnotatedMembersList(psiClass, ContextJavaBean.BEAN_JAM_KEY, CHECK_METHOD));

      return true;
    });

    return beans;
  }

  protected SpringStereotypeElement(@Nullable String anno) {
    if (anno != null) myMeta = getMeta(anno);
  }

  @NotNull
  private static synchronized JamAnnotationMeta getMeta(@NotNull String anno) {
    JamAnnotationMeta meta = annotationMetaMap.get(anno);
    if (meta == null) {
      meta = new JamAnnotationMeta(anno);
      annotationMetaMap.put(anno, meta);
    }
    return meta;
  }

  public static <Psi extends PsiModifierListOwner, Jam extends SpringStereotypeElement> void addPomTargetProducer(JamMemberMeta<Psi, Jam> classMeta) {
    classMeta.addPomTargetProducer((stereotypeElement, consumer) -> consumer.consume(stereotypeElement.getPsiTarget()));
  }

  public PsiTarget getPsiTarget() {
    final JamStringAttributeElement<String> namedAttributeValue = getNamedStringAttributeElement();
    if (namedAttributeValue == null || StringUtil.isEmptyOrSpaces(namedAttributeValue.getStringValue())) {
      return getAliasingPsiTarget();
    }

    return new JamPomTarget(this, namedAttributeValue);
  }

  private PsiTarget getAliasingPsiTarget() {
    return new AliasingPsiTarget(getPsiElement()) {
      @NotNull
      @Override
      public String getNameAlias(@NotNull String delegatePsiTargetName) {
        return StringUtil.decapitalize(delegatePsiTargetName);
      }

      @NotNull
      @Override
      public AliasingPsiTarget setAliasName(@NotNull String newAliasName) {
        AliasingPsiTargetUtil.renameTargets(this, StringUtil.capitalize(newAliasName));

        return super.setAliasName(newAliasName);
      }
    };
  }

  @Nullable
  private JamStringAttributeElement<String> getNamedStringAttributeElement() {
    return myMeta == null ? null : myMeta.getAttribute(getPsiElement(), NAME_VALUE_META);
  }

  @Nullable
  public PsiAnnotation getPsiAnnotation() {
    return myMeta == null ? null : myMeta.getAnnotation(getPsiElement());
  }

  @Nullable
  protected JamAnnotationMeta getMeta() {
    return myMeta;
  }

  @Override
  public String getBeanName() {
    assert isValid();
    final String definedName = getAnnotationDefinedBeanName();

    if (!StringUtil.isEmptyOrSpaces(definedName)) return definedName;

    return super.getBeanName();
  }

  @Nullable
  private String getAnnotationDefinedBeanName() {
    final JamStringAttributeElement<String> namedStringAttributeElement = getNamedStringAttributeElement();

    return namedStringAttributeElement == null ? null : namedStringAttributeElement.getStringValue();
  }

  @Override
  @NotNull
  public PsiElement getIdentifyingPsiElement() {
    return PomService.convertToPsi(getPsiManager().getProject(), getPsiTarget());
  }

  @Override
  public String toString() {
    final String beanName = getBeanName();
    return beanName == null ? "" : beanName;
  }
}
