// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.reflect.*;
import com.intellij.lang.properties.psi.PropertiesFile;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.jam.converters.PropertiesFileConverter;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

public class SpringJamPropertySource implements SpringPropertySource {

  public static final SemKey<SpringJamPropertySource> REPEATABLE_ANNO_JAM_KEY =
    SpringPropertySource.PROPERTY_SOURCE_JAM_KEY.subKey("SpringJamPropertySource");

  private static final JamStringAttributeMeta.Collection<Set<PropertiesFile>> VALUE_ATTR_META =
    JamAttributeMeta.collectionString("value", new PropertiesFileConverter());

  private static final JamBooleanAttributeMeta IGNORE_RESOURCE_NOT_FOUND_ATTR_META =
    JamAttributeMeta.singleBoolean(IGNORE_RESOURCE_NOT_FOUND_ATTR_NAME, false);

  static final JamAnnotationMeta ANNO_META = new JamAnnotationMeta(SpringAnnotationsConstants.PROPERTY_SOURCE);

  public static final JamClassMeta<SpringJamPropertySource> META
    = new JamClassMeta<>(null, SpringJamPropertySource.class, REPEATABLE_ANNO_JAM_KEY);

  static {
    META.addAnnotation(ANNO_META);
    ANNO_META.addAttribute(VALUE_ATTR_META);
    ANNO_META.addAttribute(IGNORE_RESOURCE_NOT_FOUND_ATTR_META);
  }

  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final PsiClass myPsiClass;

  public SpringJamPropertySource(@NotNull PsiClass psiClass) {
    myPsiClass = psiClass;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiClass);
  }

  public SpringJamPropertySource(@NotNull PsiAnnotation annotation) {
    myPsiClass = PsiTreeUtil.getParentOfType(annotation, PsiClass.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  @Override
  @NotNull
  @JamPsiConnector
  public PsiClass getPsiElement() {
    return myPsiClass;
  }

  @Override
  public boolean isPsiValid() {
    return myPsiClass.isValid();
  }

  @Override
  @Nullable
  public PsiAnnotation getAnnotation() {
    return ANNO_META.getAnnotation(getPsiElement());
  }

  @NotNull
  @Override
  public List<JamStringAttributeElement<Set<PropertiesFile>>> getLocationElements() {
    return VALUE_ATTR_META.getJam(myPsiAnnotation);
  }

  @Override
  public boolean isIgnoreResourceNotFound() {
    return IGNORE_RESOURCE_NOT_FOUND_ATTR_META.getJam(myPsiAnnotation).getValue();
  }

  @Override
  public Set<PropertiesFile> getPropertiesFiles() {
    Set<PropertiesFile> propertiesFiles = new LinkedHashSet<>();

    final List<JamStringAttributeElement<Set<PropertiesFile>>> attributeValues = VALUE_ATTR_META.getJam(myPsiAnnotation);

    for (JamStringAttributeElement<Set<PropertiesFile>> attributeElement : attributeValues) {
      final Set<PropertiesFile> value = attributeElement.getValue();
      if (value != null) propertiesFiles.addAll(value);
    }

    return propertiesFiles;
  }
}
