// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamClassAttributeMeta;
import com.intellij.jam.reflect.JamStringAttributeMeta;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.ref.AnnotationChildLink;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.aliasFor.SpringAliasFor;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.converters.PackageJamConverter;
import com.intellij.spring.model.jam.utils.filters.SpringContextFilter;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public abstract class CustomJamComponentScanBase extends SpringComponentScan {
  private final JamAnnotationMeta myMeta;
  private final AnnotationChildLink myAnnotationChildLink;
  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;

  private static final Map<String, JamAnnotationMeta> annotationMetaMap = new HashMap<>();

  protected CustomJamComponentScanBase(@NotNull String anno, @NotNull PsiClass psiClassAnchor) {
    super(psiClassAnchor);
    myAnnotationChildLink = new AnnotationChildLink(anno);
    myPsiAnnotation = myAnnotationChildLink.createChildRef(psiClassAnchor);
    myMeta = getMeta(anno);
  }

  @NotNull
  private static synchronized JamAnnotationMeta getMeta(@NotNull String anno) {
    JamAnnotationMeta meta = annotationMetaMap.get(anno);
    if (meta == null) {
      meta = new JamAnnotationMeta(anno);
      annotationMetaMap.put(anno, meta);
    }
    return meta;
  }

  @Override
  @NotNull
  public JamAnnotationMeta getAnnotationMeta() {
    return myMeta;
  }

  @Override
  @NotNull
  public PsiElementRef<PsiAnnotation> getAnnotationRef() {
    return myPsiAnnotation;
  }

  @NotNull
  @Override
  public List<JamStringAttributeMeta.Collection<Collection<PsiPackage>>> getPackageJamAttributes() {
    List<JamStringAttributeMeta.Collection<Collection<PsiPackage>>> list = new SmartList<>();

    ContainerUtil.addIfNotNull(list, getPackagesAttrMeta(BASE_PACKAGES_ATTR_NAME));
    ContainerUtil.addIfNotNull(list, getPackagesAttrMeta(VALUE_ATTR_NAME));

    return list;
  }

  @Override
  @Nullable
  protected JamClassAttributeMeta.Collection getBasePackageClassMeta() {
    SpringAliasFor aliasFor = getAliasAttribute(BASE_PACKAGE_CLASSES_ATTR_NAME);
    if (aliasFor != null) {
      return JamAttributeMeta.classCollection(aliasFor.getMethodName());
    }
    return null;
  }

  @Nullable
  private JamStringAttributeMeta.Collection<Collection<PsiPackage>> getPackagesAttrMeta(@NotNull String name) {
    SpringAliasFor aliasFor = getAliasAttribute(name);
    if (aliasFor != null) {
      return new JamStringAttributeMeta.Collection<>(aliasFor.getMethodName(), new PackageJamConverter());
    }
    return null;
  }

  @Override
  public boolean useDefaultFilters() {
    return true;
  }

  @NotNull
  @Override
  public Set<SpringContextFilter.Exclude> getExcludeContextFilters() {
    return Collections.emptySet(); // todo
  }

  @NotNull
  @Override
  public Set<SpringContextFilter.Include> getIncludeContextFilters() {
    return Collections.emptySet();  //todo
  }

  protected SpringAliasFor getAliasAttribute(@NotNull String attrName) {
    return SpringAliasForUtils.findAliasFor(getPsiElement(), getAnnotationQualifiedName(),
                                            SpringAnnotationsConstants.COMPONENT_SCAN, attrName);
  }

  protected String getAnnotationQualifiedName() {
    return myAnnotationChildLink.getAnnotationQualifiedName();
  }
}
