/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.converters;

import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiElementResolveResult;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.ResolveResult;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PackageReferenceSet;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PsiPackageReference;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.util.CommonProcessors;
import com.intellij.util.PatternUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.regex.Pattern;

public class SpringAntPatternPackageReferenceSet extends PackageReferenceSet {

  public SpringAntPatternPackageReferenceSet(@NotNull String packageName,
                                             @NotNull PsiElement element,
                                             int startInElement,
                                             @NotNull GlobalSearchScope scope) {
    super(packageName, element, startInElement, scope);
  }

  @NotNull
  @Override
  protected PsiPackageReference createReference(TextRange range, int index) {
    return new PsiPackageReference(this, range, index) {
      @Override
      protected ResolveResult @NotNull [] doMultiResolve() {
        Collection<PsiPackage> packages = new LinkedHashSet<>();
        for (PsiPackage parentPackage : getContext()) {
          packages.addAll(resolvePackages(parentPackage));
        }
        return PsiElementResolveResult.createResults(packages);
      }

      private Collection<PsiPackage> resolvePackages(@Nullable final PsiPackage context) {
        if (context == null) return Collections.emptySet();

        final String packageName = getValue();

        CommonProcessors.CollectProcessor<PsiPackage> processor = getProcessor(packageName);

        PsiPackageReference parentContextReference = myIndex > 0 ? getReferenceSet().getReference(myIndex - 1) : null;

        if (packageName.equals("*")) {
          for (PsiPackage aPackage : context.getSubPackages(getResolveScope())) {
            if (!processor.process(aPackage)) break;
          }
          return processor.getResults();
        }

        if (parentContextReference != null && parentContextReference.getValue().equals(("**"))) {
          return getSubPackages(context, processor, true);
        }

        if (packageName.equals("**")) {
          if (isLastReference()) {
            return getSubPackages(context, processor, false);
          }
          return Collections.singleton(context);
        }
        else if (packageName.contains("*") || packageName.contains("?")) {
          for (final PsiPackage subPackage : context.getSubPackages(getResolveScope())) {
            processor.process(subPackage);
          }
          return processor.getResults();
        }

        return getReferenceSet().resolvePackageName(context, packageName);
      }

      private boolean isLastReference() {
        return this.equals(getReferenceSet().getLastReference());
      }

      @NotNull
      private CommonProcessors.CollectProcessor<PsiPackage> getProcessor(@NotNull String packageName) {
        final Pattern pattern = PatternUtil.fromMask(packageName);
        return new CommonProcessors.CollectProcessor<PsiPackage>(new LinkedHashSet<>()) {
          @Override
          protected boolean accept(PsiPackage psiPackage) {
            String name = psiPackage.getName();
            return name != null && pattern.matcher(name).matches();
          }
        };
      }

      @NotNull
      private Collection<PsiPackage> getSubPackages(@NotNull PsiPackage context, CommonProcessors.CollectProcessor<PsiPackage> processor, boolean deep) {
        processSubPackages(context, processor, deep);
        return processor.getResults();
      }

      private void processSubPackages(final PsiPackage psiPackage,
                                      final CommonProcessors.CollectProcessor<PsiPackage> processor, boolean deep) {
        for (final PsiPackage subPackage : psiPackage.getSubPackages(getResolveScope())) {
          processor.process(subPackage);
          if (deep) processSubPackages(subPackage, processor, true);
        }
      }
    };
  }
}