// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.jam.JamElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Pair;
import com.intellij.patterns.ElementPattern;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiMember;
import com.intellij.semantic.SemKey;
import com.intellij.semantic.SemRegistrar;
import com.intellij.semantic.SemService;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.Consumer;
import com.intellij.util.Function;
import com.intellij.util.NotNullFunction;
import com.intellij.util.NullableFunction;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.SmartHashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static com.intellij.codeInsight.AnnotationUtil.CHECK_HIERARCHY;

public final class SpringSemContributorUtil {

  public static <T extends JamElement, Psi extends PsiMember> void registerMetaComponents(@NotNull final SemService semService,
                                                                                          @NotNull SemRegistrar registrar,
                                                                                          @NotNull ElementPattern<? extends Psi> place,
                                                                                          @NotNull final SemKey<JamMemberMeta<Psi, T>> metaKey,
                                                                                          @NotNull final SemKey<T> semKey,
                                                                                          @NotNull final NullableFunction<? super Psi, ? extends JamMemberMeta<Psi, T>> metaFunction) {
    registrar.registerSemElementProvider(metaKey, place, metaFunction);

    registrar.registerSemElementProvider(semKey, place,
                                         (NullableFunction<Psi, T>)member -> {
                                           final JamMemberMeta<Psi, T> memberMeta = semService.getSemElement(metaKey, member);
                                           return memberMeta != null ? memberMeta.createJamElement(PsiElementRef.real(member)) : null;
                                         }
    );
  }

  public static <T extends JamElement, Psi extends PsiMember> void registerRepeatableMetaComponents(@NotNull final SemService semService,
                                                                                                    @NotNull SemRegistrar registrar,
                                                                                                    @NotNull ElementPattern<? extends Psi> place,
                                                                                                    @NotNull final SemKey<JamMemberMeta<Psi, T>> metaKey,
                                                                                                    @NotNull final SemKey<T> semKey,
                                                                                                    @NotNull final NullableFunction<? super Psi, ? extends Collection<JamMemberMeta<Psi, T>>> metaFunction) {
    registrar.registerRepeatableSemElementProvider(metaKey, place, metaFunction);

    registrar.registerRepeatableSemElementProvider(semKey, place,
                                                   (NullableFunction<Psi, Collection<T>>)member -> {
                                                     final List<JamMemberMeta<Psi, T>> memberMetas =
                                                       semService.getSemElements(metaKey, member);
                                                     Collection<T> metas = new HashSet<>();
                                                     for (JamMemberMeta<Psi, T> memberMeta : memberMetas) {
                                                       ContainerUtil
                                                         .addIfNotNull(metas, memberMeta.createJamElement(PsiElementRef.real(member)));
                                                     }
                                                     return metas.isEmpty() ? null : metas;
                                                   }
    );
  }

  public static <T extends JamElement, Psi extends PsiMember> NullableFunction<Psi, JamMemberMeta<Psi, T>> createFunction(@NotNull final SemKey<T> semKey,
                                                                                                                          @NotNull final Class<? extends T> jamClass,
                                                                                                                          @NotNull final Function<? super Module, ? extends Collection<String>> annotationsGetter,
                                                                                                                          @NotNull final Function<? super Pair<String, Psi>, ? extends T> producer,
                                                                                                                          @Nullable final Consumer<? super JamMemberMeta<Psi, T>> metaConsumer) {
    return createFunction(semKey, jamClass, annotationsGetter, producer, metaConsumer, null);
  }

  /**
   * @see SpringAliasForUtils#getAnnotationMetaProducer(SemKey, JamMemberMeta[])
   */
  public static <T extends JamElement, Psi extends PsiMember> NullableFunction<Psi, JamMemberMeta<Psi, T>> createFunction(@NotNull final SemKey<T> semKey,
                                                                                                                          @NotNull final Class<? extends T> jamClass,
                                                                                                                          @NotNull final Function<? super Module, ? extends Collection<String>> annotationsGetter,
                                                                                                                          @NotNull final Function<? super Pair<String, Psi>, ? extends T> producer,
                                                                                                                          @Nullable final Consumer<? super JamMemberMeta<Psi, T>> metaConsumer,
                                                                                                                          @Nullable final NotNullFunction<? super Pair<String, Project>, ? extends JamAnnotationMeta> annotationMeta) {
    return psiMember -> {
      if (DumbService.isDumb(psiMember.getProject())) return null;
      if (psiMember instanceof PsiClass && ((PsiClass)psiMember).isAnnotationType()) return null;
      final Module module = ModuleUtilCore.findModuleForPsiElement(psiMember);
      for (final String anno : annotationsGetter.fun(module)) {
        if (AnnotationUtil.isAnnotated(psiMember, anno, CHECK_HIERARCHY)) {
          return getMeta(semKey, jamClass, producer, metaConsumer, annotationMeta, psiMember, anno);
        }
      }
      return null;
    };
  }

  public static <T extends JamElement, Psi extends PsiMember> NullableFunction<Psi, Collection<JamMemberMeta<Psi, T>>> createRepeatableFunction(
    @NotNull final SemKey<T> semKey,
    @NotNull final Class<? extends T> jamClass,
    @NotNull final Function<? super Module, ? extends Collection<String>> annotationsGetter,
    @NotNull final Function<? super Pair<String, Psi>, ? extends T> producer,
    @Nullable final Consumer<? super JamMemberMeta<Psi, T>> metaConsumer,
    @Nullable final NotNullFunction<? super Pair<String, Project>, ? extends JamAnnotationMeta> annotationMeta) {
    return psiMember -> {
      Collection<JamMemberMeta<Psi, T>> metas = new HashSet<>();
      if (DumbService.isDumb(psiMember.getProject())) return null;
      if (psiMember instanceof PsiClass && ((PsiClass)psiMember).isAnnotationType()) return null;
      final Module module = ModuleUtilCore.findModuleForPsiElement(psiMember);
      for (final String anno : annotationsGetter.fun(module)) {
        if (AnnotationUtil.isAnnotated(psiMember, anno, CHECK_HIERARCHY)) {
          final JamMemberMeta<Psi, T> meta = getMeta(semKey, jamClass, producer, metaConsumer, annotationMeta, psiMember, anno);

          metas.add(meta);
        }
      }
      return metas.isEmpty() ? null : metas;
    };
  }

  @NotNull
  private static <T extends JamElement, Psi extends PsiMember> JamMemberMeta<Psi, T> getMeta(@NotNull SemKey<T> semKey,
                                                                                             @NotNull Class<? extends T> jamClass,
                                                                                             @NotNull Function<? super Pair<String, Psi>, ? extends T> producer,
                                                                                             @Nullable Consumer<? super JamMemberMeta<Psi, T>> metaConsumer,
                                                                                             @Nullable NotNullFunction<? super Pair<String, Project>, ? extends JamAnnotationMeta> annotationMeta,
                                                                                             Psi psiMember, String anno) {
    final JamMemberMeta<Psi, T> meta = new JamMemberMeta<Psi, T>(null, jamClass, semKey) {
      @Override
      public T createJamElement(PsiElementRef<Psi> psiMemberPsiRef) {
        return producer.fun(Pair.create(anno, psiMemberPsiRef.getPsiElement()));
      }
    };
    if (metaConsumer != null) {
      metaConsumer.consume(meta);
    }
    if (annotationMeta != null) registerCustomAnnotationMeta(anno, meta, annotationMeta, psiMember.getProject());
    return meta;
  }

  private static <T extends JamElement, Psi extends PsiMember> void registerCustomAnnotationMeta(@NotNull String anno,
                                                                                                 @NotNull JamMemberMeta<Psi, T> meta,
                                                                                                 @NotNull NotNullFunction<? super Pair<String, Project>, ? extends JamAnnotationMeta> metaNotNullFunction,
                                                                                                 @NotNull Project project) {
    List<JamAnnotationMeta> annotations = meta.getAnnotations();
    for (JamAnnotationMeta annotationMeta : annotations) {
      if (anno.equals(annotationMeta.getAnnoName())) return;
    }
    meta.addAnnotation(metaNotNullFunction.fun(Pair.create(anno, project)));
  }

  /**
   * @param <T>
   * @return Consumer.
   * @see SpringStereotypeElement#addPomTargetProducer(JamMemberMeta)
   */
  public static <T extends SpringStereotypeElement, Psi extends PsiMember> Consumer<JamMemberMeta<Psi, T>> createStereotypeConsumer() {
    return SpringStereotypeElement::addPomTargetProducer;
  }

  /**
   * @param anno Annotation FQN.
   * @return Custom annotation types.
   */
  public static Function<Module, Collection<String>> getCustomMetaAnnotations(@NotNull final String anno) {
    return getCustomMetaAnnotations(anno, false);
  }

  /**
   * Returns all custom meta annotations in defined scope.
   *
   * @param anno      Annotation FQN.
   * @param withTests Whether to include annotations located in test scope.
   * @return Custom annotation types.
   */
  public static Function<Module, Collection<String>> getCustomMetaAnnotations(@NotNull final String anno, final boolean withTests) {
    return getCustomMetaAnnotations(anno, withTests, true);
  }

  /**
   * Returns all custom meta annotations in defined scope, optionally filtering custom JAM implementations.
   *
   * @param anno                           Annotation FQN.
   * @param withTests                      Whether to include annotations located in test scope.
   * @param filterCustomJamImplementations Whether to filter custom JAM implementations.
   * @return Custom annotation types.
   * @see JamCustomImplementationBean
   */
  public static Function<Module, Collection<String>> getCustomMetaAnnotations(@NotNull final String anno,
                                                                              final boolean withTests,
                                                                              final boolean filterCustomJamImplementations) {
    return new Function<Module, Collection<String>>() {

      @Override
      public Collection<String> fun(final Module module) {
        if (module == null) return Collections.emptySet();

        Collection<PsiClass> psiClasses = getAnnotationTypes(module, anno);

        final Set<String> customMetaFQNs = getJamCustomFQNs(module);

        return ContainerUtil.mapNotNull(psiClasses, psiClass -> {
          String qualifiedName = psiClass.getQualifiedName();
          if (anno.equals(qualifiedName)) return null;

          if (customMetaFQNs.contains(qualifiedName)) {
            return null;
          }
          return qualifiedName;
        });
      }

      @NotNull
      private Set<String> getJamCustomFQNs(Module module) {
        if (!filterCustomJamImplementations) return Collections.emptySet();

        final Set<String> customMetaFQNs = new SmartHashSet<>();
        for (JamCustomImplementationBean bean : JamCustomImplementationBean.EP_NAME.getExtensions()) {
          if (anno.equals(bean.baseAnnotationFqn)) {

            final String customMetaAnnotationFqn = bean.customMetaAnnotationFqn;
            customMetaFQNs.add(customMetaAnnotationFqn);

            Collection<PsiClass> customMetaClasses = getAnnotationTypes(module, customMetaAnnotationFqn);
            for (PsiClass customMetaClass : customMetaClasses) {
              ContainerUtil.addIfNotNull(customMetaFQNs, customMetaClass.getQualifiedName());
            }
          }
        }
        return customMetaFQNs;
      }

      private Collection<PsiClass> getAnnotationTypes(Module module, String anno) {
        return withTests ?
               JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildrenIncludingTests(anno) :
               JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(anno);
      }
    };
  }
}
