/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.custom;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.HashSet;

/**
 * @see com.intellij.spring.model.SpringImplicitBeansProviderBase
 */
public abstract class CustomModuleComponentsDiscoverer {

  public static final ExtensionPointName<CustomModuleComponentsDiscoverer> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.customModuleComponentsDiscoverer");

  @NotNull
  public abstract Collection<CommonSpringBean> getCustomComponents(@NotNull Module module);

  @NotNull
  public abstract String getProviderName();

  /**
   * @see CachedValueProvider.Result#getDependencyItems()
   */
  public abstract Object[] getDependencies(@NotNull Module module);

  public boolean accepts(@NotNull Module module) {
    return true;
  }

  @NotNull
  public static CommonSpringModel getCustomBeansModel(@NotNull final Module module) {
    return CachedValuesManager.getManager(module.getProject()).getCachedValue(module, () -> {
      final CustomComponentDiscovererBeansModel model =
        new CustomComponentDiscovererBeansModel(module, "") {
          @Override
          public Collection<SpringBeanPointer> getLocalBeans() {
            final Collection<CommonSpringBean> beans = new HashSet<>();
            for (CustomModuleComponentsDiscoverer discoverer : EP_NAME.getExtensionList()) {
              if (discoverer.accepts(module)) {
                beans.addAll(discoverer.getCustomComponents(module));
              }
            }
            return BeanService.getInstance().mapSpringBeans(beans);
          }
        };
      return CachedValueProvider.Result.createSingleDependency(model, ProjectRootManager.getInstance(module.getProject()));
    });
  }
}
