/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.converters;

import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Conditions;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.FileReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.FileReferenceSet;
import com.intellij.spring.model.utils.resources.SpringResourcesBuilder;
import com.intellij.spring.model.utils.resources.SpringResourcesUtil;
import com.intellij.util.Function;
import com.intellij.util.xml.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

public class SpringResourceConverter extends Converter<PsiFileSystemItem> implements CustomReferenceConverter<PsiFileSystemItem> {
  @Override
  public PsiFileSystemItem fromString(final @Nullable String s, final ConvertContext context) {
    if (s != null) {
      final DomElement domElement = context.getInvocationElement();
      if (domElement instanceof GenericAttributeValue) {
        final GenericAttributeValue attributeValue = (GenericAttributeValue)domElement;
        final PsiReference[] references = createReferences(attributeValue.getXmlAttributeValue(), attributeValue.getStringValue(),
                                                           isEndingSlashNotAllowed());
        if (references.length > 0) {
          for (int i = references.length - 1; i >= 0; i--) {
            final PsiReference reference = references[i];
            if (!(reference instanceof FileReference)) {
              continue;
            }

            PsiElement result = reference.resolve();
            if (result instanceof PsiFileSystemItem) {
              return (PsiFileSystemItem)result;
            }
          }
        }
      }
    }
    return null;
  }

  @Override
  public String toString(final @Nullable PsiFileSystemItem psiFile, final ConvertContext context) {
    return null;
  }

  @Override
  public PsiReference @NotNull [] createReferences(final GenericDomValue<PsiFileSystemItem> genericDomValue,
                                                   final PsiElement element,
                                                   final ConvertContext context) {
    return createReferences(element, genericDomValue.getStringValue(), isEndingSlashNotAllowed());
  }

  protected boolean isEndingSlashNotAllowed() {
    return true;
  }

  protected Condition<PsiFileSystemItem> getCondition() {
    return Conditions.alwaysTrue();
  }

  @Nullable
  protected Function<PsiFile, Collection<PsiFileSystemItem>> getCustomDefaultPathEvaluator(@NotNull PsiElement element, @NotNull String s) {
    return null;
  }

  private PsiReference @NotNull [] createReferences(@Nullable PsiElement element, @Nullable String s, boolean endingSlashNotAllowed) {
    if (s == null || element == null) {
      return PsiReference.EMPTY_ARRAY;
    }
    final SpringResourcesBuilder builder = SpringResourcesBuilder.create(element, s).
      fromRoot(StringUtil.startsWithChar(s, '/')).
      filter(getCondition()).
      endingSlashNotAllowed(endingSlashNotAllowed).
      customDefaultPathEvaluator(getCustomDefaultPathEvaluator(element, s));

    return SpringResourcesUtil.getInstance().getReferences(builder);
  }

  public static class Allowed extends SpringResourceConverter {
    @Override
    protected boolean isEndingSlashNotAllowed() {
      return false;
    }
  }

  public static class Directory extends SpringResourceConverter {

    @Override
    protected Condition<PsiFileSystemItem> getCondition() {
      return FileReferenceSet.DIRECTORY_FILTER;
    }
  }
}
