// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.converters;

import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.codeInspection.LocalQuickFixProvider;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClassType;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.SpringModelVisitorUtils;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.converters.fixes.bean.SpringBeanResolveQuickFixManager;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.spring.model.xml.beans.Beans;
import com.intellij.util.ReflectionUtil;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.GenericDomValue;
import com.intellij.util.xml.converters.DelimitedListConverter;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public class SpringBeanListConverter extends DelimitedListConverter<SpringBeanPointer> {

  public SpringBeanListConverter() {
    super(SpringCommonUtils.SPRING_DELIMITERS);
  }

  @Override
  public boolean canResolveTo(Class<? extends PsiElement> elementClass) {
    return ReflectionUtil.isAssignable(SpringBeanPointer.class, elementClass);
  }

  @Override
  @Nullable
  protected SpringBeanPointer convertString(final @Nullable String s, final ConvertContext context) {
    if (s == null) return null;

    final CommonSpringModel model = SpringConverterUtil.getSpringModel(context);
    if (model == null) return null;

    return SpringModelSearchers.findBean(model, s);
  }

  @Override
  @Nullable
  protected String toString(@Nullable final SpringBeanPointer springBeanPointer) {
    return springBeanPointer == null ? null : springBeanPointer.getName();
  }

  @Override
  @Nullable
  protected PsiElement resolveReference(final SpringBeanPointer springBeanPointer,
                                        final ConvertContext context) {
    return springBeanPointer == null ? null : springBeanPointer.getPsiElement();
  }

  @Override
  protected String getUnresolvedMessage(final String value) {
    return SpringApiBundle.message("model.bean.error.message", value);
  }

  @Override
  protected Object[] getReferenceVariants(final ConvertContext context,
                                          GenericDomValue<? extends List<SpringBeanPointer>> genericDomValue) {
    final CommonSpringModel model = SpringConverterUtil.getSpringModel(context);
    if (model == null) return EMPTY_ARRAY;

    List<SpringBeanPointer> variants = new ArrayList<>();

    final DomSpringBean currentBean = SpringConverterUtil.getCurrentBean(context);

    final Collection<SpringBeanPointer> allBeans = getVariantBeans(context, model);
    for (SpringBeanPointer pointer : allBeans) {
      if (pointer.isReferenceTo(currentBean)) continue;

      for (String string : SpringModelVisitorUtils.getAllBeanNames(model, pointer)) {
        if (StringUtil.isNotEmpty(string)) {
          variants.add(pointer.derive(string));
        }
      }
    }

    final List<SpringBeanPointer> existingBeans = genericDomValue.getValue();
    if (existingBeans != null) {
      for (Iterator<SpringBeanPointer> it = variants.iterator(); it.hasNext(); ) {
        final CommonSpringBean variant = it.next().getSpringBean();
        for (SpringBeanPointer existing : existingBeans) {
          if (existing.isReferenceTo(variant)) {
            it.remove();
            break;
          }
        }
      }
    }

    List<LookupElement> result = new ArrayList<>(variants.size());
    for (final SpringBeanPointer pointer : variants) {
      ContainerUtil.addIfNotNull(result, SpringConverterUtil.createCompletionVariant(pointer));
    }

    return result.toArray();
  }

  protected Collection<SpringBeanPointer> getVariantBeans(ConvertContext convertContext,
                                                          @NotNull CommonSpringModel model) {
    final List<PsiClassType> requiredBeanTypeClasses = SpringConverterUtil.getRequiredBeanTypeClasses(convertContext);
    if (!requiredBeanTypeClasses.isEmpty()) {
      final CommonSpringBean currentBean = SpringConverterUtil.getCurrentBeanCustomAware(convertContext);
      return SpringConverterUtil.getSmartVariants(currentBean, requiredBeanTypeClasses, model);
    }
    return model.getAllCommonBeans();
  }

  @Override
  @NotNull
  protected PsiReference createPsiReference(final PsiElement element,
                                            final int start,
                                            final int end,
                                            final ConvertContext context,
                                            final GenericDomValue<List<SpringBeanPointer>> genericDomValue,
                                            final boolean delimitersOnly) {
    return new MyFixableReference(element, getTextRange(genericDomValue, start, end), context, genericDomValue, delimitersOnly);
  }


  private final class MyFixableReference extends MyPsiReference implements LocalQuickFixProvider {

    private final Beans beans;

    private MyFixableReference(final PsiElement element,
                               final TextRange range,
                               final ConvertContext context,
                               final GenericDomValue<List<SpringBeanPointer>> genericDomValue,
                               final boolean delimitersOnly) {
      super(element, range, context, genericDomValue, delimitersOnly);
      this.beans = genericDomValue.getParentOfType(Beans.class, false);
    }

    @Override
    public LocalQuickFix[] getQuickFixes() {
      return SpringBeanResolveQuickFixManager.getInstance()
                                             .getQuickFixes(myContext, beans, getValue(),
                                                            SpringConverterUtil.getRequiredBeanTypeClasses(myContext));
    }
  }
}
