// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.converters;

import com.intellij.psi.ElementManipulators;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.PsiReference;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.model.utils.SpringReferenceUtils;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.CustomReferenceConverter;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

/**
 * @author Dmitry Avdeev
 */
public class PackageListConverter extends Converter<Collection<PsiPackage>> implements CustomReferenceConverter {

  @Override
  @NotNull
  public Collection<PsiPackage> fromString(@Nullable @NonNls final String s, final ConvertContext context) {
    if (s == null) {
      return Collections.emptyList();
    }
    if (context.getXmlElement() == null) {
      return Collections.emptyList();
    }

    if (SpringConverterUtil.containsPatternReferences(s)) {
      final PsiReference[] references = createReferences((GenericDomValue)context.getInvocationElement(), context.getXmlElement(), context);
      return SpringConverterUtil.getPsiPackages(references);
    }

    return SpringConverterUtil.getPackages(s, getDelimiters(), context.getPsiManager().getProject());
  }

  @Override
  public String toString(@Nullable final Collection<PsiPackage> psiPackages, final ConvertContext context) {
    return null;
  }

  @Override
  public PsiReference @NotNull [] createReferences(final GenericDomValue genericDomValue, final PsiElement element, final ConvertContext context) {
    final String text = genericDomValue.getStringValue();
    if (text == null) {
      return PsiReference.EMPTY_ARRAY;
    }
    GlobalSearchScope scope = context.getSearchScope();
    return SpringReferenceUtils.getPsiPackagesReferences(element, text, ElementManipulators.getOffsetInElement(element), getDelimiters(),
                                                         scope != null ? scope : GlobalSearchScope.allScope(element.getProject()));
  }

  protected String getDelimiters() {
    return ",; \n\t";
  }
}
