// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Comparing;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.qualifiers.SpringJamQualifier;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import gnu.trove.THashSet;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import org.jetbrains.annotations.Nullable;

public final class SpringQualifierComparator {

  public static boolean compareQualifiers(@Nullable SpringQualifier one, @Nullable final SpringQualifier two) {
    if (one == null || two == null) return false;
    if (!Comparing.equal(one.getQualifierType(), two.getQualifierType())) return false;
    if (!Objects.equals(one.getQualifierValue(), two.getQualifierValue())) return false;
    final List<? extends QualifierAttribute> list1 = one.getQualifierAttributes();
    final int size1 = list1.size();
    final List<? extends QualifierAttribute> list2 = two.getQualifierAttributes();
    final int size2 = list2.size();
    if (size1 != size2) return false;
    if (size1 == 0) return true;
    final THashSet<QualifierAttribute> set = new THashSet<>(QualifierAttribute.HASHING_STRATEGY);
    set.addAll(list1);
    return set.containsAll(list2);
  }

  public static boolean compareInheritorQualifier(@Nullable SpringQualifier childrenQualifier,
                                                  @Nullable final SpringQualifier baseQualifier,
                                                  @Nullable Module module) {
    if (childrenQualifier instanceof SpringJamQualifier &&
        baseQualifier instanceof SpringJamQualifier) {
      if (Comparing.equal(childrenQualifier.getQualifierType(), baseQualifier.getQualifierType())) return false;

      if (module == null) return false;

      final PsiClass baseType = baseQualifier.getQualifierType();
      final PsiClass childrenType = childrenQualifier.getQualifierType();
      if (baseType != null && childrenType != null) {
        final String baseAnnoQualifiedName = baseType.getQualifiedName();
        if (baseAnnoQualifiedName != null) {
          final Collection<PsiClass> children =
            JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(baseAnnoQualifiedName);
          final PsiAnnotation definingMetaAnnotation =
            SpringAliasForUtils.findDefiningMetaAnnotation(childrenType, baseAnnoQualifiedName, children);
          if (definingMetaAnnotation != null) {
            return compareQualifiers(new SpringJamQualifier(definingMetaAnnotation, null), baseQualifier);
          }
        }
      }
    }

    return false;
  }
}
