// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.gutter.groups;

import com.intellij.codeInsight.daemon.GutterIconNavigationHandler;
import com.intellij.codeInsight.daemon.RelatedItemLineMarkerInfo;
import com.intellij.codeInsight.navigation.NavigationGutterIconBuilder;
import com.intellij.codeInsight.navigation.NavigationGutterIconRenderer;
import com.intellij.navigation.GotoRelatedItem;
import com.intellij.openapi.editor.markup.GutterIconRenderer;
import com.intellij.psi.PsiElement;
import com.intellij.util.ConstantFunction;
import com.intellij.util.NotNullFunction;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class SpringGutterIconBuilder<T> extends NavigationGutterIconBuilder<T> {

  public static final GutterIconRenderer.Alignment DEFAULT_GUTTER_ICON_ALIGNMENT = GutterIconRenderer.Alignment.LEFT;

  public static SpringGutterIconBuilder<PsiElement> createBuilder(@NotNull Icon icon) {
    return new SpringGutterIconBuilder<>(icon, DEFAULT_PSI_CONVERTOR, PSI_GOTO_RELATED_ITEM_PROVIDER);
  }

  public static <T> SpringGutterIconBuilder<T> createBuilder(@NotNull Icon icon,
                                                             @NotNull NotNullFunction<? super T, ? extends Collection<? extends PsiElement>> converter,
                                                             @Nullable NotNullFunction<? super T, ? extends Collection<? extends GotoRelatedItem>> gotoRelatedItemProvider) {
    return new SpringGutterIconBuilder<>(icon, converter, gotoRelatedItemProvider);
  }

  private SpringGutterIconBuilder(@NotNull Icon icon,
                                  @NotNull NotNullFunction<? super T, ? extends Collection<? extends PsiElement>> converter,
                                  @Nullable NotNullFunction<? super T, ? extends Collection<? extends GotoRelatedItem>> gotoRelatedItemProvider) {
    super(icon, converter, gotoRelatedItemProvider);
  }


  @NotNull
  public RelatedItemLineMarkerInfo<PsiElement> createSpringGroupLineMarkerInfo(@NotNull PsiElement element) {
    NavigationGutterIconRenderer renderer = createGutterIconRenderer(element.getProject());
    return new SpringGroupRelatedItemLineMarkerInfo(element, renderer, renderer, ()->computeGotoTargets());
  }

  @NotNull
  public RelatedItemLineMarkerInfo<PsiElement> createSpringRelatedMergeableLineMarkerInfo(@NotNull PsiElement element) {
    NavigationGutterIconRenderer renderer = createGutterIconRenderer(element.getProject());
    return new SpringRelatedMergeableLineMarkerInfo(element, renderer, renderer, ()->computeGotoTargets());
  }


  public static final class CustomNavigationHandlerBuilder<T> extends SpringGutterIconBuilder<T> {

    @NotNull
    private final Icon myIcon;

    @NotNull
    private final String myTooltipText;

    @NotNull
    private final GutterIconNavigationHandler<PsiElement> myNavigationHandler;

    private @Nullable String myElementPresentation;

    private @NotNull Collection<GotoRelatedItem> myAdditionalGotoRelatedItems = Collections.emptyList();

    private CustomNavigationHandlerBuilder(@NotNull Icon icon,
                                           @NotNull String tooltipText,
                                           @NotNull GutterIconNavigationHandler<PsiElement> navigationHandler,
                                           @NotNull NotNullFunction<T, Collection<? extends PsiElement>> converter,
                                           @Nullable NotNullFunction<T, Collection<? extends GotoRelatedItem>> gotoRelatedItemProvider) {
      super(icon, converter, gotoRelatedItemProvider);
      myIcon = icon;
      myTooltipText = tooltipText;
      myNavigationHandler = navigationHandler;
    }

    /**
     * Creates builder for custom navigation popup (EXPERIMENTAL API).
     *
     * @param icon                    Gutter icon.
     * @param tooltipText             Gutter icon tooltip.
     * @param navigationHandler       Provides custom navigation popup.
     * @param gotoRelatedItemProvider (Optional) For supporting "Goto Related Symbols".
     * @param <T>                     Type.
     * @return Builder instance.
     */
    public static <T> CustomNavigationHandlerBuilder<T> createBuilder(@NotNull Icon icon,
                                                                      @NotNull String tooltipText,
                                                                      @NotNull GutterIconNavigationHandler<PsiElement> navigationHandler,
                                                                      @Nullable NotNullFunction<T, Collection<? extends GotoRelatedItem>> gotoRelatedItemProvider) {
      return new CustomNavigationHandlerBuilder<>(icon, tooltipText, navigationHandler,
                                                  new ConstantFunction<>(Collections.emptyList()),
                                                  gotoRelatedItemProvider);
    }

    public CustomNavigationHandlerBuilder<T> withElementPresentation(@NotNull String elementPresentation) {
      myElementPresentation = elementPresentation;
      return this;
    }

    public @NotNull CustomNavigationHandlerBuilder<T> withAdditionalGotoRelatedItems(@NotNull Collection<GotoRelatedItem> items) {
      myAdditionalGotoRelatedItems = items;
      return this;
    }

    @Override
    @NotNull
    public RelatedItemLineMarkerInfo<PsiElement> createSpringRelatedMergeableLineMarkerInfo(@NotNull PsiElement element) {
      return new SpringRelatedMergeableLineMarkerInfo(element, myIcon, myTooltipText, myNavigationHandler,
                                                      () -> {
                                                        List<GotoRelatedItem> result = new ArrayList<>(computeGotoTargets());
                                                        result.addAll(myAdditionalGotoRelatedItems);
                                                        return result;
                                                      },
                                                      myElementPresentation);
    }
  }
}
