/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.facet.searchers;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.module.Module;
import com.intellij.psi.search.GlobalSearchScope;
import org.jetbrains.annotations.NotNull;

/**
 * Allows filtering scope when searching for suitable configuration files.
 * <p/>
 * Can be used to filter (framework-)specific folders/filters/.. which should not be available for user configuration.
 */
public abstract class ConfigSearcherScopeModifier {

  private static final ExtensionPointName<ConfigSearcherScopeModifier> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.configSearcherScopeModifier");

  /**
   * Modifies the given scope.
   *
   * @param module        Current module.
   * @param originalScope Scope to modify.
   * @return Modified scope or originalScope if not applicable.
   */
  @NotNull
  public abstract GlobalSearchScope modifyScope(@NotNull Module module, @NotNull GlobalSearchScope originalScope);

  public static GlobalSearchScope runModifiers(@NotNull Module module,
                                               @NotNull GlobalSearchScope originalScope) {
    GlobalSearchScope scope = originalScope;
    for (ConfigSearcherScopeModifier modifier : EP_NAME.getExtensions()) {
      scope = modifier.modifyScope(module, scope);
    }
    return scope;
  }
}
