/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.facet;

import com.intellij.facet.FacetManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleManager;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.project.ProjectManager;
import com.intellij.openapi.vfs.impl.LightFilePointer;
import com.intellij.openapi.vfs.pointers.VirtualFilePointer;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;

import javax.swing.*;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public abstract class SpringAutodetectedFileSet extends SpringFileSet {

  private final Icon myIcon;

  protected SpringAutodetectedFileSet(String id, String name, SpringFacet parent,
                                      final Icon icon) {
    super(id, name, parent);
    myIcon = icon;
  }

  @Override
  public final Icon getIcon() {
    return myIcon;
  }

  @Override
  public final boolean isAutodetected() {
    return true;
  }

  @Override
  public final void setAutodetected(boolean autodetected) {
    throw new UnsupportedOperationException();
  }

  @Override
  protected final VirtualFilePointer createVirtualFilePointer(@NotNull String url) {
    return new LightFilePointer(url);
  }

  @Override
  public final void setActiveProfiles(@NotNull Set<String> activeProfiles) {
    super.setActiveProfiles(activeProfiles);
    getFacet().getConfiguration().setActiveProfilesForAutodetected(getId(), activeProfiles);
  }

  public static void refreshAutodetectedFileSets() {
    SpringFileSetService fileSetService = SpringFileSetService.getInstance();
    for (Project project : ProjectManager.getInstance().getOpenProjects()) {
      for (Module module : ModuleManager.getInstance(project).getModules()) {
        SpringFacet facet = SpringFacet.getInstance(module);
        if (facet != null) {
          Set<SpringFileSet> fileSets = new HashSet<>(facet.getFileSets());
          List<String> providerSets = ContainerUtil.map(fileSetService.getModelProviderSets(facet),
                                                        SpringFileSet::getId);
          boolean modified =
            ContainerUtil.retainAll(fileSets, fileSet -> !fileSet.isAutodetected() || providerSets.contains(fileSet.getId()));
          if (modified) {
            facet.removeFileSets();
            for (SpringFileSet fileSet : fileSets) {
              facet.addFileSet(fileSet);
            }
          }
          else {
            facet.getConfiguration().setModified();
          }
          // update FacetConfig listeners (Spring Toolwindow)
          FacetManager.getInstance(module).facetConfigurationChanged(facet);
        }
      }
    }
  }
}
