// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.dom.converters;

import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInsight.lookup.LookupElementBuilder;
import com.intellij.openapi.util.AtomicNotNullLazyValue;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.CharsetToolkit;
import com.intellij.spring.SpringApiBundle;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.ResolvingConverter;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.nio.charset.Charset;
import java.util.Collection;
import java.util.Set;

/**
 * @author Yann C&eacute;bron
 */
public class CharsetConverter extends ResolvingConverter.StringConverter {

  private final AtomicNotNullLazyValue<Set<String>> charSets = new AtomicNotNullLazyValue<Set<String>>() {
    @NotNull
    @Override
    protected Set<String> compute() {
      return ContainerUtil.map2Set(CharsetToolkit.getAvailableCharsets(), Charset::name);
    }
  };

  @Override
  public String fromString(final String s, final ConvertContext convertContext) {
    if (StringUtil.isEmpty(s)) {
      return null;
    }

    return charSets.getValue().contains(s) ? s : null;
  }

  @Nullable
  @Override
  public LookupElement createLookupElement(String s) {
    return LookupElementBuilder.create(s).withCaseSensitivity(false);
  }

  @NotNull
  @Override
  public Collection<String> getVariants(final ConvertContext convertContext) {
    return charSets.getValue();
  }

  @Override
  public String getErrorMessage(@Nullable String s, ConvertContext context) {
    return SpringApiBundle.message("cannot.resolve.charset", s);
  }
}