// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.dom;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.util.ArrayUtilRt;
import com.intellij.util.xml.DomFileDescription;
import com.intellij.util.xml.reflect.DomExtensionsRegistrar;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Allows registration of custom namespace(s) in Spring DOM.
 */
public abstract class SpringCustomNamespaces {

  public static final ExtensionPointName<SpringCustomNamespaces> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.customNamespaces");

  public abstract NamespacePolicies getNamespacePolicies();

  /**
   * Convenience method to register namespace DOM elements.
   * <p/>
   * Alternatively, one can use custom {@link com.intellij.util.xml.reflect.DomExtender}.
   *
   * @param registrar Registrar instance.
   * @see CustomNamespaceRegistrar
   */
  public void registerExtensions(DomExtensionsRegistrar registrar) {
  }

  /**
   * Returns DOM model version for all namespaces registered in this EP.
   * <p/>
   * Must be incremented on any Spring model related change
   * (e.g. changing {@link com.intellij.spring.model.xml.BeanType} or
   * linked {@link com.intellij.spring.model.xml.BeanTypeProvider},
   * custom implementation of {@link com.intellij.spring.model.xml.DomSpringBeanImpl#getBeanName()})
   * to update all related indexes.
   *
   * @return Model version.
   */
  public int getModelVersion() {
    return 0;
  }

  /**
   * Returns DOM stub version for all namespaces registered in this EP.
   * <p/>
   * Must be incremented on any change of using {@link com.intellij.util.xml.Stubbed} in related DOM.
   *
   * @return 0 if no stubs are used (default).
   * @see com.intellij.util.xml.Stubbed
   */
  public int getStubVersion() {
    return 0;
  }


  /**
   * @see DomFileDescription#registerNamespacePolicy(String, String...)
   */
  public static class NamespacePolicies {
    private final Map<String, List<String>> policies = new HashMap<>();

    public static NamespacePolicies simple(String key, String namespace) {
      NamespacePolicies policies = new NamespacePolicies();
      return policies.add(key, namespace);
    }

    public NamespacePolicies add(String key, String... namespaces) {
      policies.put(key, Arrays.asList(namespaces));
      return this;
    }

    /**
     * Checks whether given namespace is registered with this policy.
     *
     * @param namespace
     * @return true if registered.
     */
    public boolean isRegistered(String namespace) {
      for (List<String> namespaces : policies.values()) {
        if (namespaces.contains(namespace)) {
          return true;
        }
      }
      return false;
    }

    void process(DomFileDescription domFileDescription) {
      for (Map.Entry<String, List<String>> entry : policies.entrySet()) {
        domFileDescription.registerNamespacePolicy(entry.getKey(), ArrayUtilRt.toStringArray(entry.getValue()));
      }
    }
  }
}
