// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.util.CommonProcessors;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Set;

public abstract class NoCacheProcessor<InParams extends SpringModelSearchParameters> {

  protected NoCacheProcessor() {
  }

  @NotNull
  protected abstract Collection<SpringBeanPointer> findPointers(@NotNull InParams parameters);

  @Nullable
  protected abstract SpringBeanPointer findFirstPointer(@NotNull InParams parameters);

  public boolean process(InParams params,
                         final Processor<SpringBeanPointer> processor,
                         final Set<String> activeProfiles) {
    if (processor instanceof CommonProcessors.FindFirstProcessor) {
      return processBeansInActiveProfile(processor, findFirstPointer(params), activeProfiles);
    }

    for (SpringBeanPointer pointer : findPointers(params)) {
      if (!processBeansInActiveProfile(processor, pointer, activeProfiles)) return false;
    }
    return true;
  }

  private static boolean processBeansInActiveProfile(Processor<SpringBeanPointer> processor,
                                                     @Nullable SpringBeanPointer pointer,
                                                     Set<String> activeProfiles) {
    if (pointer == null) return true;
    if (ContainerUtil.isEmpty(activeProfiles)) return processor.process(pointer);

    if (!SpringProfileUtils.isInActiveProfiles(pointer.getSpringBean(), activeProfiles)) return true;

    return processor.process(pointer);
  }
}
