// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.jam.JamService;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.*;
import com.intellij.psi.impl.compiled.ClsFileImpl;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.PackageScope;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.psi.xml.XmlFile;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringLocalModelFactory;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.contexts.model.graph.LocalModelDependency;
import com.intellij.spring.contexts.model.graph.LocalModelDependencyType;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.spring.model.jam.SpringJamModel;
import com.intellij.spring.model.jam.javaConfig.ContextJavaBean;
import com.intellij.spring.model.jam.javaConfig.SpringJavaBean;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import com.intellij.spring.model.jam.stereotype.SpringComponentScan;
import com.intellij.spring.model.jam.stereotype.SpringConfiguration;
import com.intellij.spring.model.jam.utils.SpringJamUtils;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.ArrayUtil;
import com.intellij.util.Processor;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.intellij.jam.JamService.CHECK_DEEP;
import static com.intellij.jam.JamService.CHECK_METHOD;

public abstract class LocalAnnotationModel extends AbstractSimpleLocalModel<PsiClass> {
  @NotNull private final PsiClass myClass;
  @NotNull protected final Module myModule;
  @NotNull protected final Set<String> myActiveProfiles;
  @NotNull private final CachedValue<Collection<ContextJavaBean>> myLocalContextBeansCachedValue;

  public LocalAnnotationModel(@NotNull PsiClass aClass, @NotNull Module module, @NotNull Set<String> activeProfiles) {
    myClass = aClass;
    myModule = module;
    myActiveProfiles = Collections.unmodifiableSet(activeProfiles);
    myLocalContextBeansCachedValue = CachedValuesManager.getManager(myClass.getProject()).createCachedValue(() -> {
      List<ContextJavaBean> beans = JamService.getJamService(myClass.getProject())
        .getAnnotatedMembersList(myClass, ContextJavaBean.BEAN_JAM_KEY, CHECK_METHOD | CHECK_DEEP);
      Set<PsiFile> dependencies = ContainerUtil.newHashSet(myClass.getContainingFile());
      for (ContextJavaBean bean : beans) {
        if (bean.isValid()) {
          ContainerUtil.addIfNotNull(dependencies, bean.getContainingFile());
        }
      }
      return CachedValueProvider.Result.create(beans, dependencies);
    });
  }

  @Override
  @NotNull
  public PsiClass getConfig() {
    return myClass;
  }

  @Override
  public final Collection<SpringBeanPointer> getLocalBeans() {
    Collection<CommonSpringBean> allBeans = new SmartList<>();

    ContainerUtil.addIfNotNull(allBeans, getBeanForClass(myClass));
    ContainerUtil.addAllNotNull(allBeans, SpringProfileUtils
      .filterBeansInActiveProfiles(myLocalContextBeansCachedValue.getValue(), getActiveProfiles()));

    return BeanService.getInstance().mapSpringBeans(allBeans);
  }

  @Nullable
  private static CommonSpringBean getBeanForClass(@NotNull PsiClass aClass) {
    return CachedValuesManager.getCachedValue(aClass, () -> {
      CommonSpringBean commonSpringBean =
        JamService.getJamService(aClass.getProject()).getJamElement(JamPsiMemberSpringBean.PSI_MEMBER_SPRING_BEAN_JAM_KEY, aClass);
      if (commonSpringBean == null) {
        if (!aClass.isInterface() && !aClass.hasModifierProperty(PsiModifier.ABSTRACT) &&
            SpringCommonUtils.isSpringBeanCandidateClass(aClass)) {
          commonSpringBean = new CustomSpringComponent(aClass);
        }
      }
      return CachedValueProvider.Result.createSingleDependency(commonSpringBean, aClass);
    });
  }

  @Override
  public Set<CommonSpringModel> getRelatedModels() {
    Set<CommonSpringModel> models = new LinkedHashSet<>();

    ContainerUtil.addAllNotNull(models, getRelatedLocalModels());
    ContainerUtil.addAllNotNull(models, getCachedPackageScanModel());
    ContainerUtil.addAllNotNull(models, getCachedInnerStaticClassConfigurations());
    ContainerUtil.addAllNotNull(models, getCustomDiscoveredBeansModel());

    return models;
  }

  @NotNull
  private Set<CommonSpringModel> getCachedPackageScanModel() {
    return CachedValuesManager.getManager(getConfig().getProject()).getCachedValue(this, () -> CachedValueProvider.Result
      .create(getPackageScanModels(this), getOutsideModelDependencies(this)));
  }


  @NotNull
  protected Set<CommonSpringModel> getPackageScanModels(@NotNull LocalAnnotationModel localModel) {
    Set<CommonSpringModel> models = new LinkedHashSet<>();
    Module module = localModel.getModule();
    for (SpringBeansPackagesScan scan : localModel.getPackagesScans()) {
      models.add(new SpringComponentScanModel<>(module, scan, localModel.getActiveProfiles()));
    }

    return models;
  }

  @NotNull
  private Set<CommonSpringModel> getCachedInnerStaticClassConfigurations() {
    return CachedValuesManager.getManager(getConfig().getProject()).getCachedValue(this, () -> CachedValueProvider.Result
      .create(getInnerStaticClassConfigurations(getConfig()), getOutsideModelDependencies(this)));
  }

  private Set<CommonSpringModel> getInnerStaticClassConfigurations(@NotNull PsiClass config) {
    return getInnerStaticClassModels(config, aClass -> getLocalAnnotationModel(aClass));
  }

  public static Set<CommonSpringModel> getInnerStaticClassModels(@NotNull PsiClass config,
                                                                 Function<? super PsiClass, ? extends CommonSpringModel> mapper) {
    return Arrays.stream(config.getAllInnerClasses())
      .filter(psiClass -> psiClass.hasModifierProperty(PsiModifier.STATIC)
                          && JamService.getJamService(psiClass.getProject()).getJamElement(SpringConfiguration.JAM_KEY, psiClass) != null)
      .map(mapper)
      .collect(Collectors.toSet());
  }

  @Override
  @NotNull
  public Module getModule() {
    return myModule;
  }

  @NotNull
  @Override
  public Set<String> getProfiles() {
    Set<String> allProfiles = new LinkedHashSet<>();
    SpringConfiguration configuration = getConfiguration();
    if (configuration != null) {
      allProfiles.addAll(configuration.getProfile().getNames());
      for (SpringJavaBean bean : configuration.getBeans()) {
        allProfiles.addAll(bean.getProfile().getNames());
      }
    }
    return allProfiles;
  }

  @Nullable
  private SpringConfiguration getConfiguration() {
    return JamService.getJamService(myClass.getProject()).getJamElement(SpringConfiguration.JAM_KEY, myClass);
  }

  @NotNull
  @Override
  public List<SpringBeansPackagesScan> getPackagesScans() {
    List<SpringBeansPackagesScan> packageScans = new SmartList<>();

    packageScans.addAll(SpringJamUtils.getInstance().getBeansPackagesScan(myClass));
    for (PsiClass superClass : InheritanceUtil.getSuperClasses(myClass)) {
      if (CommonClassNames.JAVA_LANG_OBJECT.equals(superClass.getQualifiedName())) continue;
      packageScans.addAll(SpringJamUtils.getInstance().getBeansPackagesScan(superClass));
    }
    return packageScans;
  }

  @NotNull
  @Override
  public Set<String> getActiveProfiles() {
    return myActiveProfiles;
  }

  @NotNull
  @Override
  public Set<Pair<LocalModel, LocalModelDependency>> getDependentLocalModels() {
    return
      CachedValuesManager.getManager(getConfig().getProject()).getCachedValue(this, () -> {
        final Module module = getModule();
        Set<Pair<LocalModel, LocalModelDependency>> models = new HashSet<>();
        if (!module.isDisposed()) {
          SpringJamUtils.getInstance().processImportedResources(myClass, pair -> {
            for (XmlFile xmlFile : pair.first) {
              addNotNullModel(models,
                              SpringLocalModelFactory.getInstance().getOrCreateLocalXmlModel(xmlFile, module, getActiveProfiles()),
                              LocalModelDependency.create(LocalModelDependencyType.IMPORT, pair.second));
            }
            return true;
          }, myModule);

          SpringJamUtils.getInstance().processImportedClasses(myClass, pair -> {
            addNotNullModel(models,
                            getLocalAnnotationModel(pair.first),
                            LocalModelDependency.create(LocalModelDependencyType.IMPORT, pair.second));
            return true;
          });

          processScannedConfigurations(myClass, pair -> {
            if (module.isDisposed()) return false;

            GlobalSearchScope moduleScope = GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module);
            final GlobalSearchScope searchScope = moduleScope.intersectWith(PackageScope.packageScope(pair.first, true));
            final List<SpringConfiguration> configurations = SpringJamModel.getModel(module).getConfigurations(searchScope);
            for (SpringConfiguration configuration : configurations) {
              if (SpringCommonUtils.isSpringBeanCandidateClass(configuration.getPsiElement()) &&
                  SpringProfileUtils.isInActiveProfiles(configuration, getActiveProfiles())) {
                PsiElement identifyingElementForDependency = pair.second;
                final LocalModelDependency dependency =
                  LocalModelDependency.create(LocalModelDependencyType.COMPONENT_SCAN, identifyingElementForDependency == null
                                                                                       ? pair.first
                                                                                       : identifyingElementForDependency);
                addNotNullModel(models, getLocalAnnotationModel(configuration), dependency);
              }
            }
            return true;
          });

          SpringJamUtils.getInstance().processCustomAnnotations(myClass, enableAnnotation -> {
            addNotNullModel(models, getLocalAnnotationModel(enableAnnotation.getFirst()), enableAnnotation.getSecond());
            return true;
          });

          SpringJamUtils.getInstance().processCustomDependentLocalModels(this, (model, dependency) -> {
            addNotNullModel(models, model, dependency);
            return true;
          });
        }
        final Set<Object> dependencies = new LinkedHashSet<>();
        ContainerUtil.addAll(dependencies, getOutsideModelDependencies(this));
        dependencies.addAll(models.stream().map(pair -> pair.first.getConfig()).collect(Collectors.toSet()));

        return CachedValueProvider.Result.create(models, ArrayUtil.toObjectArray(dependencies));
      });
  }

  @Nullable
  protected LocalAnnotationModel getLocalAnnotationModel(@NotNull PsiClass aClass) {
    return SpringLocalModelFactory.getInstance().getOrCreateLocalAnnotationModel(aClass, getModule(), getActiveProfiles());
  }

  @Nullable
  private LocalAnnotationModel getLocalAnnotationModel(@NotNull SpringConfiguration configuration) {
    return SpringLocalModelFactory.getInstance().getOrCreateLocalAnnotationModel(configuration.getPsiElement(),
                                                                                 getModule(),
                                                                                 getActiveProfiles());
  }


  protected boolean processScannedConfigurations(@NotNull PsiClass classToProcess,
                                                 @NotNull Processor<Pair<PsiPackage, ? extends PsiElement>> processor) {
    final List<? extends SpringBeansPackagesScan> scans = SpringJamUtils.getInstance().getBeansPackagesScan(classToProcess);
    for (SpringBeansPackagesScan packagesScan : scans) {
      if (!processPackagesScan(processor, packagesScan)) return false;
    }
    return true;
  }

  protected boolean processPackagesScan(@NotNull Processor<Pair<PsiPackage, ? extends PsiElement>> processor,
                                        SpringBeansPackagesScan packagesScan) {
    if (packagesScan instanceof SpringComponentScan) {
      SpringComponentScan componentScan = (SpringComponentScan)packagesScan;
      if (!componentScan.processPsiPackages(processor)) {
        return false;
      }
    }
    else {
      for (PsiPackage aPackage : packagesScan.getPsiPackages()) {
        if (!processor.process(Pair.create(aPackage, packagesScan.getIdentifyingPsiElement()))) {
          return false;
        }
      }
    }
    return true;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof LocalAnnotationModel)) return false;

    LocalAnnotationModel model = (LocalAnnotationModel)o;

    if (!myClass.equals(model.myClass)) return false;
    if (!myModule.equals(model.myModule)) return false;
    if (!SpringProfileUtils.profilesAsString(myActiveProfiles).equals(SpringProfileUtils.profilesAsString(model.myActiveProfiles))) {
      return false;
    }


    return true;
  }

  @Override
  public int hashCode() {
    int result = myClass.hashCode();
    result = 31 * result + myModule.hashCode();
    result = 31 * result + (SpringProfileUtils.profilesAsString(myActiveProfiles).hashCode());
    return result;
  }

  @Override
  protected Collection<Object> getCachingProcessorsDependencies() {
    Collection<Object> dependencies = new HashSet<>();

    Collections.addAll(dependencies, SpringModificationTrackersManager.getInstance(myClass.getProject()).getOuterModelsDependencies());
    // all non-libs files in class hierarchy
    Collections.addAll(dependencies, Arrays.stream(myClass.getSupers()).map(aClass -> aClass.getContainingFile())
      .filter(psiFile -> psiFile != null && !(psiFile instanceof ClsFileImpl)).toArray());

    return dependencies;
  }

  @NotNull
  @Override
  public final List<SpringBeanPointer> findQualified(@NotNull SpringQualifier qualifier) {
    return findLocalBeansByQualifier(this, qualifier);  // don't use caching map here (like LocalXmlModel) as found bean could be located in super class.
  }
}