/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.chooser;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.NlsSafe;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Determines context for handling Spring configuration file in editor.
 */
public class SpringContextDescriptor {

  /**
   * PsiFile: cached context descriptor.
   */
  public static final Key<SpringContextDescriptor> KEY = Key.create("SpringContextDescriptor.KEY");

  /**
   * Use all available filesets.
   */
  public static final SpringContextDescriptor ALL_CONTEXTS = new SpringContextDescriptor(null, "all.contexts", "All Contexts");

  /**
   * Use local file only.
   */
  public static final SpringContextDescriptor LOCAL_CONTEXT = new SpringContextDescriptor(null, "local.file.context", "Local File");

  /**
   * Default context.
   */
  public static final SpringContextDescriptor DEFAULT = LOCAL_CONTEXT;

  @Nullable
  private final Module myModule;

  private final String myId;
  private final String myName;

  public SpringContextDescriptor(@Nullable Module module,
                                 @NotNull String id,
                                 @NotNull String name) {
    myModule = module;
    myId = id;
    myName = name;
  }

  @Nullable
  public Module getModule() {
    return myModule;
  }

  public String getId() {
    return myId;
  }

  public @NlsSafe String getName() {
    return myName;
  }

  /**
   * @return {@code true} if instance is not user-chosen context
   */
  public boolean isPredefinedContext() {
    return this == LOCAL_CONTEXT ||
           this == ALL_CONTEXTS;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    SpringContextDescriptor that = (SpringContextDescriptor)o;

    if (!myId.equals(that.myId)) return false;
    if (myModule != null ? !myModule.equals(that.myModule) : that.myModule != null) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = myModule != null ? myModule.hashCode() : 0;
    result = 31 * result + myId.hashCode();
    return result;
  }

  @NotNull
  public String getQualifiedName() {
    return getId();
  }
}
