/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring;

import com.intellij.openapi.module.Module;
import com.intellij.spring.contexts.model.visitors.CommonSpringModelVisitorContext;
import com.intellij.spring.contexts.model.visitors.SpringModelVisitors;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.util.Processor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

public interface CommonSpringModel {

  /**
   * Spring models:
   * - have connections/links/associations to other models (via imports/component-scans/implicit fileset configurations/auto-configurations/etc.)
   * - contain other models {@link com.intellij.spring.contexts.model.SpringModel}
   *
   * @return related models
   * @see SpringModelVisitors#visitRelatedModels(CommonSpringModel, CommonSpringModelVisitorContext)
   */
  default Set<CommonSpringModel> getRelatedModels() {
    return Collections.emptySet();
  }

  /**
   * @param params    Search parameters.
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @see SpringModelSearchParameters#canSearch()
   */
  boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params, @NotNull Processor<SpringBeanPointer> processor);

  /**
   * @param params    Search parameters.
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @see SpringModelSearchParameters#canSearch()
   */
  boolean processByName(@NotNull SpringModelSearchParameters.BeanName params, @NotNull Processor<SpringBeanPointer> processor);

  /**
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   */
  default boolean processAllBeans(@NotNull Processor<SpringBeanPointer> processor) {return true;}

  /**
   * NOTE: Expensive operation. Consider using {@code process...()} methods instead.
   *
   * @return All beans.
   */
  @NotNull
  Collection<SpringBeanPointer> getAllCommonBeans();

  /**
   * Returns the associated module.
   *
   * @return Module containing this model or {@code null} if model is not bound to a specific module (e.g. Project-global).
   */
  @Nullable
  Module getModule();

  @Nullable
  Set<String> getActiveProfiles();
}
