/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.model;

import com.intellij.database.DatabaseBundle;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.containers.JBIterable;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.NotNull;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import static java.lang.String.format;

/**
 * Kind o database object.
 * @author Leonid Bushuev from JetBrains
 */
public class ObjectKind implements Comparable<ObjectKind> {
  private static final AtomicInteger ourOrderNumCounter = new AtomicInteger(0);
  public static final Map<String, ObjectKind> ourKinds = new ConcurrentHashMap<>();

  public static final ObjectKind NONE                 = new ObjectKind("NONE", DatabaseBundle.messagePointer("ObjectKind.none"));
  public static final ObjectKind ROOT                 = new ObjectKind("ROOT", DatabaseBundle.messagePointer("ObjectKind.root"));
  public static final ObjectKind DATABASE             = new ObjectKind("DATABASE", DatabaseBundle.messagePointer("ObjectKind.database"));
  public static final ObjectKind SCHEMA               = new ObjectKind("SCHEMA", DatabaseBundle.messagePointer("ObjectKind.schema"));
  public static final ObjectKind SEQUENCE             = new ObjectKind("SEQUENCE", DatabaseBundle.messagePointer("ObjectKind.sequence"));
  public static final ObjectKind CLUSTER              = new ObjectKind("CLUSTER", DatabaseBundle.messagePointer("ObjectKind.cluster"));
  public static final ObjectKind OBJECT_TYPE          = new ObjectKind("OBJECT TYPE", DatabaseBundle.messagePointer("ObjectKind.object.type"));
  public static final ObjectKind COLLECTION_TYPE      = new ObjectKind("COLLECTION TYPE", DatabaseBundle.messagePointer("ObjectKind.collection.type"));
  public static final ObjectKind TABLE_TYPE           = new ObjectKind("TABLE TYPE", DatabaseBundle.messagePointer("ObjectKind.table.type"));
  public static final ObjectKind ALIAS_TYPE           = new ObjectKind("ALIAS TYPE", DatabaseBundle.messagePointer("ObjectKind.alias.type"));
  public static final ObjectKind TABLE                = new ObjectKind("TABLE", DatabaseBundle.messagePointer("ObjectKind.table"));
  public static final ObjectKind MAT_LOG              = new ObjectKind("MATERIALIZED LOG", DatabaseBundle.messagePointer("ObjectKind.materialized.log"));
  public static final ObjectKind MAT_VIEW             = new ObjectKind("MATERIALIZED VIEW", DatabaseBundle.messagePointer("ObjectKind.materialized.view"));
  public static final ObjectKind VIEW                 = new ObjectKind("VIEW", DatabaseBundle.messagePointer("ObjectKind.view"));
  public static final ObjectKind PACKAGE              = new ObjectKind("PACKAGE", DatabaseBundle.messagePointer("ObjectKind.package"));
  public static final ObjectKind BODY                 = new ObjectKind("BODY", DatabaseBundle.messagePointer("ObjectKind.body"));
  public static final ObjectKind ROUTINE              = new ObjectKind("ROUTINE", DatabaseBundle.messagePointer("ObjectKind.routine"));
  public static final ObjectKind METHOD               = new ObjectKind("METHOD", DatabaseBundle.messagePointer("ObjectKind.method"));
  public static final ObjectKind OPERATOR             = new ObjectKind("OPERATOR", DatabaseBundle.messagePointer("ObjectKind.operator"));
  public static final ObjectKind OBJECT_ATTRIBUTE     = new ObjectKind("OBJECT ATTRIBUTE", DatabaseBundle.messagePointer("ObjectKind.object.attribute"));
  public static final ObjectKind COLUMN               = new ObjectKind("COLUMN", DatabaseBundle.messagePointer("ObjectKind.column"));
  public static final ObjectKind INDEX                = new ObjectKind("INDEX", DatabaseBundle.messagePointer("ObjectKind.index"));
  public static final ObjectKind KEY                  = new ObjectKind("KEY", DatabaseBundle.messagePointer("ObjectKind.key"));
  public static final ObjectKind FOREIGN_KEY          = new ObjectKind("FOREIGN KEY", DatabaseBundle.messagePointer("ObjectKind.foreign.key"));
  public static final ObjectKind CHECK                = new ObjectKind("CHECK", DatabaseBundle.messagePointer("ObjectKind.check"));
  public static final ObjectKind DEFAULT              = new ObjectKind("DEFAULT", DatabaseBundle.messagePointer("ObjectKind.default"));
  public static final ObjectKind RULE                 = new ObjectKind("RULE", DatabaseBundle.messagePointer("ObjectKind.rule"));
  public static final ObjectKind TRIGGER              = new ObjectKind("TRIGGER", DatabaseBundle.messagePointer("ObjectKind.trigger"));
  public static final ObjectKind ARGUMENT             = new ObjectKind("ARGUMENT", DatabaseBundle.messagePointer("ObjectKind.argument"));
  public static final ObjectKind VARIABLE             = new ObjectKind("VARIABLE", DatabaseBundle.messagePointer("ObjectKind.variable"));
  public static final ObjectKind SYNONYM              = new ObjectKind("SYNONYM", DatabaseBundle.messagePointer("ObjectKind.synonym"));
  public static final ObjectKind DB_LINK              = new ObjectKind("DBLINK", DatabaseBundle.messagePointer("ObjectKind.dblink"));
  public static final ObjectKind VIRTUAL_TABLE        = new ObjectKind("VIRTUAL TABLE", DatabaseBundle.messagePointer("ObjectKind.virtual.table"));
  public static final ObjectKind COLLATION            = new ObjectKind("COLLATION", DatabaseBundle.messagePointer("ObjectKind.collation"));
  public static final ObjectKind SCRIPT               = new ObjectKind("SCRIPT", DatabaseBundle.messagePointer("ObjectKind.script"));
  public static final ObjectKind TABLESPACE           = new ObjectKind("TABLESPACE", DatabaseBundle.messagePointer("ObjectKind.tablespace"));
  public static final ObjectKind DATA_FILE            = new ObjectKind("DATA FILE", DatabaseBundle.messagePointer("ObjectKind.data.file"));
  public static final ObjectKind ROLE                 = new ObjectKind("ROLE", DatabaseBundle.messagePointer("ObjectKind.role"));
  public static final ObjectKind USER                 = new ObjectKind("USER", DatabaseBundle.messagePointer("ObjectKind.user"));
  public static final ObjectKind CONNECTION           = new ObjectKind("CONNECTION", DatabaseBundle.messagePointer("ObjectKind.connection"));
  public static final ObjectKind FOREIGN_DATA_WRAPPER = new ObjectKind("FOREIGN DATA WRAPPER", DatabaseBundle.messagePointer("ObjectKind.foreign.data.wrapper"));
  public static final ObjectKind SERVER               = new ObjectKind("SERVER", DatabaseBundle.messagePointer("ObjectKind.server"));
  public static final ObjectKind USER_MAPPING         = new ObjectKind("USER MAPPING", DatabaseBundle.messagePointer("ObjectKind.user.mapping"));
  public static final ObjectKind FOREIGN_TABLE        = new ObjectKind("FOREIGN TABLE", DatabaseBundle.messagePointer("ObjectKind.foreign.table"));
  public static final ObjectKind EXTERNAL_SCHEMA      = new ObjectKind("EXTERNAL SCHEMA", DatabaseBundle.messagePointer("ObjectKind.external.schema"));
  public static final ObjectKind SCHEDULED_EVENT      = new ObjectKind("SCHEDULED EVENT", DatabaseBundle.messagePointer("ObjectKind.scheduled.event"));
  public static final ObjectKind ACCESS_METHOD        = new ObjectKind("ACCESS METHOD", DatabaseBundle.messagePointer("ObjectKind.access.method"));
  public static final ObjectKind AGGREGATE            = new ObjectKind("AGGREGATE", DatabaseBundle.messagePointer("ObjectKind.aggregate"));
  public static final ObjectKind EXCEPTION            = new ObjectKind("EXCEPTION", DatabaseBundle.messagePointer("ObjectKind.exception"));
  public static final ObjectKind EXTENSION            = new ObjectKind("EXTENSION", DatabaseBundle.messagePointer("ObjectKind.extension"));
  public static final ObjectKind PROJECTION           = new ObjectKind("PROJECTION", DatabaseBundle.messagePointer("ObjectKind.projection"));
  public static final ObjectKind MACRO                = new ObjectKind("MACRO", DatabaseBundle.messagePointer("ObjectKind.macro"));
  public static final ObjectKind PARTITION            = new ObjectKind("PARTITION", DatabaseBundle.messagePointer("ObjectKind.partition"));
  public static final ObjectKind WAREHOUSE            = new ObjectKind("WAREHOUSE", DatabaseBundle.messagePointer("ObjectKind.warehouse"));
  public static final ObjectKind FORMAT               = new ObjectKind("FORMAT", DatabaseBundle.messagePointer("ObjectKind.format"));
  public static final ObjectKind INDEX_EXTENSION      = new ObjectKind("INDEX EXTENSION", DatabaseBundle.messagePointer("ObjectKind.index.extension"));
  public static final ObjectKind INDEX_SEARCH_METHOD  = new ObjectKind("SEARCH METHOD", DatabaseBundle.messagePointer("ObjectKind.search.method"));
  public static final ObjectKind CONSTANT             = new ObjectKind("CONSTANT", DatabaseBundle.messagePointer("ObjectKind.constant"));
  public static final ObjectKind PERIOD               = new ObjectKind("PERIOD", DatabaseBundle.messagePointer("ObjectKind.period"));
  public static final ObjectKind LANGUAGE             = new ObjectKind("LANGUAGE", DatabaseBundle.messagePointer("ObjectKind.language"));
  public static final ObjectKind LOGIN                = new ObjectKind("LOGIN", DatabaseBundle.messagePointer("ObjectKind.login"));
  public static final ObjectKind OPERATOR_CLASS       = new ObjectKind("OPERATOR CLASS", DatabaseBundle.messagePointer("ObjectKind.operator.class"));
  public static final ObjectKind OPERATOR_FAMILY      = new ObjectKind("OPERATOR FAMILY", DatabaseBundle.messagePointer("ObjectKind.operator.family"));

  public static final ObjectKind UNKNOWN_OBJECT       = new ObjectKind("UNKNOWN OBJECT" , DatabaseBundle.messagePointer("ObjectKind.unknown.object"), Integer.MAX_VALUE);

  private static final int ourLastDatabaseKind = ourOrderNumCounter.get();

  private final String myName;
  private final Supplier<@Nls String> myPresentableName;
  private final int myOrderNum;

  private final String myCode;

  public ObjectKind(@NotNull String name, Supplier<@Nls String> presentableName) {
    this(name, presentableName, ourOrderNumCounter.getAndIncrement());
  }

  private ObjectKind(@NotNull String name, Supplier<@Nls String> presentableName, int orderNum) {
    myPresentableName = presentableName;
    assert name.length() > 0;
    assert orderNum >= 0;
    myName = name;
    myCode = StringUtil.toLowerCase(myName).replace(' ', '-');
    myOrderNum = orderNum;
    ourKinds.putIfAbsent(myCode, this);
  }

  public String name() {
    return myName;
  }

  /**
   * Returns the formal code, that can be used to write it into a file.
   * This code also XML-friendly, starts with a letter and can contain
   * letters, digits and dashes.
   * @return the formal code.
   */
  public String code() {
    return myCode;
  }

  @Nls
  public String getPresentableName() {
    return myPresentableName.get();
  }

  /**
   * Returns the order number influences on order of this kind of elements in the script.
   * @return non-negative integer order number.
   */
  public int getOrder() {
    return myOrderNum;
  }

  @Override
  public String toString() {
    return code();
  }

  @Override
  public int compareTo(@NotNull ObjectKind that) {
    if (this == that) return 0;
    if (this.myOrderNum < that.myOrderNum) return -1;
    if (this.myOrderNum > that.myOrderNum) return +1;
    throw new IllegalStateException(format("Uncomparable object kinds: %s and %s", this.code(), that.code()));
  }

  public static @NotNull JBIterable<ObjectKind> getDatabaseKinds() {
    return JBIterable.from(ourKinds.values()).filter(ObjectKind::isDatabaseKind);
  }

  public static boolean isDatabaseKind(@NotNull ObjectKind k) {
    return k.myOrderNum < ourLastDatabaseKind;
  }
}
