/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.jam;

import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiLanguageInjectionHost;
import com.intellij.psi.PsiReference;
import com.intellij.psi.UastReferenceRegistrar;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * @author peter
 */
public abstract class JamConverter<T> {

  public static final JamConverter<String> DUMMY_CONVERTER = new JamConverter<String>() {
    @Override
    public String fromString(@Nullable String s, JamStringAttributeElement<String> context) {
      return s;
    }

    @Override
    public String toString(@Nullable String s, JamElement context) {
      return s;
    }
  };

  public static final JamConverter<Boolean> BOOLEAN_CONVERTER = new JamConverter<Boolean>() {
    @Nullable
    @Override
    public Boolean fromString(@Nullable String s, JamStringAttributeElement<Boolean> context) {
      return s == null ? null : Boolean.valueOf(s);
    }
  };

  @Nullable
  public abstract T fromString(@Nullable String s, JamStringAttributeElement<T> context);

  @Nullable
  public String toString(@Nullable T s, JamElement context) {
    throw new UnsupportedOperationException("toString() not supported for " + getClass());
  }

  /**
   * Supports creating references in literals if they are used as constants inside annotations.
   * The default implementation delegates to {@link #createReferences(JamStringAttributeElement<T>, PsiLanguageInjectionHost)} and does not
   * support injection "by usage".
   *
   * @param context       annotation attribute, may contain expressions and should not be used for references
   * @param injectionHost place where reference should be injected
   * @param usagePsi      child of annotation attribute,
   *                      either the same as injectionHost or reference expression that resolves to expression containing injectionHost
   * @see UastReferenceRegistrar#registerReferenceProviderByUsage
   */
  public PsiReference @NotNull [] createReferences(@NotNull JamStringAttributeElement<T> context,
                                                   @NotNull PsiLanguageInjectionHost injectionHost,
                                                   @NotNull PsiElement usagePsi) {
    if (injectionHost != usagePsi) return PsiReference.EMPTY_ARRAY;

    return createReferences(context, injectionHost);
  }

  /**
   * @see #createReferences(JamStringAttributeElement, PsiLanguageInjectionHost, PsiElement)
   */
  public PsiReference @NotNull [] createReferences(@NotNull JamStringAttributeElement<T> context,
                                                   @SuppressWarnings("unused") @NotNull PsiLanguageInjectionHost injectionHost) {
    return createReferences(context);
  }

  /**
   * @deprecated Override {@link #createReferences(JamStringAttributeElement, PsiLanguageInjectionHost, PsiElement)} instead.
   */
  @Deprecated
  @ApiStatus.ScheduledForRemoval(inVersion = "2020.3")
  public PsiReference @NotNull [] createReferences(@NotNull JamStringAttributeElement<T> context) {
    return PsiReference.EMPTY_ARRAY;
  }
}
