/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.persistence.util;

import com.intellij.facet.FacetFinder;
import com.intellij.facet.FacetManager;
import com.intellij.facet.FacetType;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleManager;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.persistence.PersistenceHelper;
import com.intellij.persistence.facet.PersistenceFacet;
import com.intellij.persistence.facet.PersistenceFacetType;
import com.intellij.persistence.model.*;
import com.intellij.persistence.roles.PersistenceClassRole;
import com.intellij.persistence.roles.PersistenceClassRoleEnum;
import com.intellij.persistence.roles.PersistenceRoleHolder;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.ProjectScope;
import com.intellij.psi.util.*;
import com.intellij.util.*;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.GenericValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;

import static com.intellij.psi.util.CachedValueProvider.Result.create;

/**
 * @author Gregory.Shrago
 */
public final class PersistenceCommonUtil {
  private PersistenceCommonUtil() { }

  @NotNull
  public static List<PersistenceFacet> getAllPersistenceFacets(@NotNull final Project project) {
    return CachedValuesManager.getManager(project).getCachedValue(project, () -> {
      List<PersistenceFacet> result = new ArrayList<>();
      for (Module module : ModuleManager.getInstance(project).getModules()) {
        result.addAll(getAllPersistenceFacets(module));
      }
      return create(result, getPersistenceModificationTrackerDependencies(project));
    });
  }

  public static Object[] getPersistenceModificationTrackerDependencies(@NotNull Project project) {
    Set<Object> dependencies = new HashSet<>();
    dependencies.add(PsiModificationTracker.MODIFICATION_COUNT);
    dependencies.add(ProjectRootManager.getInstance(project));
    dependencies.add(DumbService.getInstance(project).getModificationTracker());
    dependencies.addAll(Arrays.asList(getFacetCacheDependencies(project)));
    return ArrayUtil.toObjectArray(dependencies);
  }

  @NotNull
  public static List<PersistenceFacet> getAllPersistenceFacets(@NotNull final Module module) {
    return Arrays.stream(FacetManager.getInstance(module).getAllFacets())
      .filter(PersistenceFacet.class::isInstance)
      .map(PersistenceFacet.class::cast)
      .collect(Collectors.toList());
  }

  private static final Key<CachedValue<List<PersistenceFacet>>> MODULE_PERSISTENCE_FACETS = Key.create("MODULE_PERSISTENCE_FACETS");

  @NotNull
  public static List<PersistenceFacet> getAllPersistenceFacetsWithDependencies(@NotNull final Module module) {
    if (module.isDisposed()) return Collections.emptyList();
    CachedValue<List<PersistenceFacet>> cachedValue =
      module.getUserData(MODULE_PERSISTENCE_FACETS);
    if (cachedValue == null) {
      cachedValue = CachedValuesManager.getManager(module.getProject()).createCachedValue(() -> {
        final Set<Module> modules = new HashSet<>();
        ContainerUtil.addAll(modules, JamCommonUtil.getAllDependentModules(module));
        ContainerUtil.addAll(modules, JamCommonUtil.getAllModuleDependencies(module));
        final Set<PersistenceFacet> facets = new HashSet<>();
        for (Module depModule : modules) {
          facets.addAll(getAllPersistenceFacets(depModule));
        }
        return new CachedValueProvider.Result<>(
          new ArrayList<>(facets),
          getFacetCacheDependencies(module.getProject()));
      }, false);
      module.putUserData(MODULE_PERSISTENCE_FACETS, cachedValue);
    }
    return cachedValue.getValue();
  }

  public static Object[] getFacetCacheDependencies(@NotNull Project project) {
    Set<Object> dependencies = new HashSet<>();
    dependencies.add(ProjectRootManager.getInstance(project));
    FacetFinder facetFinder = FacetFinder.getInstance(project);
    for (FacetType<?, ?> facetType : FacetType.EP_NAME.getExtensionList()) {
      if (facetType instanceof PersistenceFacetType) {
        dependencies.add(facetFinder.getAllFacetsOfTypeModificationTracker(facetType.getId()));
      }
    }
    return dependencies.toArray();
  }

  public static PersistenceModelBrowser createSameUnitsModelBrowser(@Nullable final PsiElement sourceElement) {
    final PsiClass sourceClass;
    final Set<PersistencePackage> unitSet;
    if (sourceElement == null || (sourceClass = PsiTreeUtil.getParentOfType(sourceElement, PsiClass.class, false)) == null) {
      unitSet = null;
    }
    else {
      unitSet = getAllPersistenceUnits(sourceClass, new HashSet<>());
    }
    return createUnitsAndTypeMapper(unitSet);
  }

  public static PersistenceModelBrowser createSameUnitsModelBrowser(@Nullable final DomElement sourceDom) {
    final Set<PersistencePackage> unitSet;
    final DomElement rootElement;
    if (sourceDom == null || !((rootElement = DomUtil.getFileElement(sourceDom).getRootElement()) instanceof PersistenceMappings)) {
      unitSet = null;
    }
    else {
      unitSet = new HashSet<>(PersistenceHelper.getHelper().getSharedModelBrowser().getPersistenceUnits((PersistenceMappings)rootElement));
    }
    return createUnitsAndTypeMapper(unitSet);
  }

  public static PersistenceModelBrowser createUnitsAndTypeMapper(@Nullable final Set<PersistencePackage> unitSet) {
    return PersistenceHelper.getHelper().createModelBrowser().setRoleFilter(role -> {
      final PersistentObject object = role.getPersistentObject();
      final PersistenceClassRoleEnum roleType = role.getType();
      return roleType != PersistenceClassRoleEnum.ENTITY_LISTENER &&
             object != null &&
             (unitSet == null || unitSet.contains(role.getPersistenceUnit()));
    });
  }

  public static PersistenceModelBrowser createFacetAndUnitModelBrowser(final PersistenceFacet facet,
                                                                       final PersistencePackage unit,
                                                                       @Nullable final PersistenceClassRoleEnum type) {
    return PersistenceHelper.getHelper().createModelBrowser().setRoleFilter(role -> {
      final PersistentObject object = role.getPersistentObject();
      return object != null && (type == null || role.getType() == type) && (unit == null || unit.equals(role.getPersistenceUnit())) &&
             (facet == null || facet.equals(role.getFacet()));
    });
  }

  @Nullable
  public static PsiType getTargetEntityType(final PsiMember psiMember) {
    return getTargetEntityType(PropertyUtilBase.getPropertyType(psiMember));
  }

  @Nullable
  public static PsiType getTargetEntityType(final PsiType type) {
    return getTypeInfo(type).getValueType();
  }

  public static <T extends Collection<PersistencePackage>> T getAllPersistenceUnits(@Nullable final PsiClass sourceClass,
                                                                                    @NotNull final T result) {
    for (PersistenceClassRole role : getPersistenceRoles(sourceClass)) {
      ContainerUtil.addIfNotNull(result, role.getPersistenceUnit());
    }
    return result;
  }

  @NotNull
  public static <V extends PersistenceMappings> Collection<V> getDomEntityMappings(
    final Class<V> mappingsClass, final PersistencePackage unit, final PersistenceFacet facet) {
    final Set<V> result = new HashSet<>();
    for (PersistenceMappings mappings : facet.getDefaultEntityMappings(unit)) {
      if (ReflectionUtil.isAssignable(mappingsClass, mappings.getClass())) {
        result.add((V)mappings);
      }
    }
    for (GenericValue<V> value : unit.getModelHelper().getMappingFiles(mappingsClass)) {
      ContainerUtil.addIfNotNull(result, value.getValue());
    }
    return result;
  }

  public static boolean isSameTable(final TableInfoProvider table1, final TableInfoProvider table2) {
    if (table1 == null || table2 == null) return false;
    final String name1 = table1.getTableName().getValue();
    return StringUtil.isNotEmpty(name1) &&
           Objects.equals(name1, table2.getTableName().getValue()) &&
           Objects.equals(table1.getSchema().getValue(), table2.getSchema().getValue()) &&
           Objects.equals(table1.getCatalog().getValue(), table2.getCatalog().getValue());
  }

  public static String getUniqueId(final PsiElement psiElement) {
    final VirtualFile virtualFile = psiElement == null ? null : PsiUtilCore.getVirtualFile(psiElement);
    return virtualFile == null ? "" : virtualFile.getUrl();
  }

  public static String getMultiplicityString(final boolean optional, final boolean many) {
    final String first = (optional ? "0" : "1");
    final String last = (many ? "*" : "1");
    return first.equals(last) ? first : first + ".." + last;
  }

  public static <T, V extends Collection<T>> V mapPersistenceRoles(final V result,
                                                                   final Project project,
                                                                   final PersistenceFacet facet,
                                                                   final PersistencePackage unit,
                                                                   final Function<? super PersistenceClassRole, ? extends T> mapper) {
    for (PersistenceClassRole role : getPersistenceRoles(project)) {
      if ((facet == null || facet == role.getFacet()) && (unit == null || unit == role.getPersistenceUnit())) {
        ContainerUtil.addIfNotNull(result, mapper.fun(role));
      }
    }
    return result;
  }

  public static boolean haveCorrespondingMultiplicity(final PersistentRelationshipAttribute a1, final PersistentRelationshipAttribute a2) {
    return a1.getAttributeModelHelper().getRelationshipType().corresponds(a2.getAttributeModelHelper().getRelationshipType());
  }


  @NotNull
  public static JavaTypeInfo getTypeInfo(final PsiType type) {
    return getTypeInfo(type, null);
  }

  @NotNull
  public static JavaTypeInfo getTypeInfo(final PsiType type, @Nullable PsiClass convertClass) {
    if (type instanceof PsiArrayType) {
      return new JavaTypeInfo(JavaContainerType.ARRAY, type, convertClass, ((PsiArrayType)type).getComponentType());
    }
    final PsiClassType.ClassResolveResult classResolveResult = type instanceof PsiClassType ? ((PsiClassType)type).resolveGenerics() : null;
    final PsiClass psiClass = classResolveResult == null ? null : classResolveResult.getElement();
    if (psiClass == null) return new JavaTypeInfo(null, type, convertClass);
    final PsiManager manager = psiClass.getManager();
    final GlobalSearchScope scope = ProjectScope.getAllScope(manager.getProject());
    for (JavaContainerType collectionType : JavaContainerType.values()) {
      if (collectionType == JavaContainerType.ARRAY) continue;
      final PsiClass aClass = JavaPsiFacade.getInstance(manager.getProject()).findClass(collectionType.getJavaBaseClassName(), scope);
      if (aClass != null && (manager.areElementsEquivalent(aClass, psiClass) || psiClass.isInheritor(aClass, true))) {
        final PsiSubstitutor superClassSubstitutor =
          TypeConversionUtil.getSuperClassSubstitutor(aClass, psiClass, classResolveResult.getSubstitutor());
        final JavaTypeInfo result = new JavaTypeInfo(collectionType, type, convertClass, ArrayUtil.reverseArray(
          ContainerUtil.map2Array(aClass.getTypeParameters(), PsiType.class,
                                  (NullableFunction<PsiTypeParameter, PsiType>)psiTypeParameter -> superClassSubstitutor
                                    .substitute(psiTypeParameter))));
        if (result.containerType == JavaContainerType.MAP && result.parameters.length != 2
            || JavaContainerType.isCollection(result.containerType) && result.parameters.length != 1) {
          return new JavaTypeInfo(null, type);
        }
        return result;
      }
    }
    return new JavaTypeInfo(null, type, convertClass);
  }

  public static Query<PersistentObject> queryPersistentObjects(final PersistenceMappings mappings) {
    return new ExecutorsQuery<>(mappings, Collections.singletonList(
      (queryParameters, consumer) -> {
        if (!ContainerUtil.process(queryParameters.getModelHelper().getPersistentEntities(), consumer)) return false;
        if (!ContainerUtil.process(queryParameters.getModelHelper().getPersistentSuperclasses(), consumer)) return false;
        if (!ContainerUtil.process(queryParameters.getModelHelper().getPersistentEmbeddables(), consumer)) return false;
        return true;
      }));
  }

  @Nullable
  public static PsiClass getTargetClass(final PersistentRelationshipAttribute attribute) {
    final GenericValue<PsiClass> classValue = attribute.getTargetEntityClass();
    final PsiClass targetClass;
    if (classValue.getStringValue() != null) {
      targetClass = classValue.getValue();
    }
    else {
      final PsiType entityType = getTargetEntityType(attribute.getPsiMember());
      targetClass = entityType instanceof PsiClassType ? ((PsiClassType)entityType).resolve() : null;
    }
    return targetClass;
  }

  @Nullable
  public static PsiClass getTargetClass(final PersistentEmbeddedAttribute attribute) {
    final GenericValue<PsiClass> classValue = attribute.getTargetEmbeddableClass();
    final PsiClass targetClass;
    if (classValue.getStringValue() != null) {
      targetClass = classValue.getValue();
    }
    else {
      final PsiType entityType = PropertyUtilBase.getPropertyType(attribute.getPsiMember());
      targetClass = entityType instanceof PsiClassType ? ((PsiClassType)entityType).resolve() : null;
    }
    return targetClass;
  }

  @Nullable
  public static PsiType getPrimaryKeyClass(@NotNull PersistentEntityBase persistentObject, final PersistenceModelBrowser browser) {
    final ArrayList<PersistentAttribute> idAttrs = new ArrayList<>();
    for (PersistentObject object : browser.queryPersistentObjectHierarchy(persistentObject)) {
      if (object instanceof PersistentEntityBase) {
        final PsiClass idClass = ((PersistentEntityBase)object).getIdClassValue().getValue();
        if (idClass != null) return JavaPsiFacade.getElementFactory(idClass.getProject()).createType(idClass);
        for (PersistentAttribute idAttr : object.getObjectModelHelper().getAttributes()) {
          if (idAttr.getAttributeModelHelper().isIdAttribute()) {
            idAttrs.add(idAttr);
          }
        }
      }
    }
    return idAttrs.size() == 1 ? idAttrs.get(0).getPsiType() : null;
  }

  public static PersistenceClassRole[] getPersistenceRoles(final PsiClass psiClass) {
    if (psiClass == null) return PersistenceClassRole.EMPTY_ARRAY;

    return CachedValuesManager.getCachedValue(psiClass, () -> {
      return create(calculatePersistenceRoles(psiClass),
                    getPersistenceModificationTrackerDependencies(psiClass.getProject()));
    });
  }

  public static PersistenceClassRole[] getPersistenceRoles(@NotNull Project project) {
    return CachedValuesManager.getManager(project)
      .getCachedValue(project, () -> create(calculatePersistenceRoles(project), getPersistenceModificationTrackerDependencies(project)));
  }

  private static PersistenceClassRole[] calculatePersistenceRoles(PsiClass psiClass) {
    CommonProcessors.CollectProcessor<PersistenceClassRole> collectProcessor = new CommonProcessors.CollectProcessor<>();
    final Module module = ModuleUtilCore.findModuleForPsiElement(psiClass);
    if (module != null) {
      PersistenceRoleHolder.getInstance(psiClass.getProject()).processModuleRoles(module, psiClass, collectProcessor);
    }
    else {
      PersistenceRoleHolder.getInstance(psiClass.getProject()).processAllRoles(psiClass, collectProcessor);
    }
    return ContainerUtil.toArray(collectProcessor.getResults(), PersistenceClassRole.ARRAY_FACTORY);
  }

  private static PersistenceClassRole[] calculatePersistenceRoles(@NotNull Project project) {
    CommonProcessors.CollectProcessor<PersistenceClassRole> collectProcessor = new CommonProcessors.CollectProcessor<>();
    PersistenceRoleHolder.getInstance(project).processAllRoles(collectProcessor);

    return ContainerUtil.toArray(collectProcessor.getResults(), PersistenceClassRole.ARRAY_FACTORY);
  }

  @NotNull
  public static Collection<? extends PsiMember> getAttributePsiMembers(final PersistentAttribute attributeBase) {
    final String name = attributeBase.getName().getValue();
    if (StringUtil.isEmpty(name)) return Collections.emptyList();
    final PsiMember member = attributeBase.getPsiMember();
    if (member == null) return Collections.emptyList();
    final PsiClass psiClass = member.getContainingClass();
    if (psiClass == null) return Collections.emptyList();

    final PsiMethod getter = PropertyUtilBase.findPropertyGetter(psiClass, name, false, false);
    final PsiMethod setter = PropertyUtilBase.findPropertySetter(psiClass, name, false, false);
    PsiField field = psiClass.findFieldByName(name, true);
    if (member == getter && field == null) {
      field = PropertyUtilBase.findPropertyField(psiClass, name, false);
      if (field == null) {
        final PsiCodeBlock body = getter.getBody();
        if (body != null) {
          for (PsiStatement statement : body.getStatements()) {
            if (statement instanceof PsiReturnStatement) {
              final PsiReturnStatement returnStatement = (PsiReturnStatement)statement;
              final PsiExpression returnExpression = returnStatement.getReturnValue();
              if (returnExpression instanceof PsiReferenceExpression) {
                final PsiReferenceExpression referenceExpression = (PsiReferenceExpression)returnExpression;
                final PsiElement psiElement = referenceExpression.resolve();
                if (psiElement instanceof PsiField) {
                  field = (PsiField)psiElement;
                }
              }
            }
          }
        }
      }
    }
    final ArrayList<PsiMember> result = new ArrayList<>(3);
    ContainerUtil.addIfNotNull(result, field);
    ContainerUtil.addIfNotNull(result, setter);
    ContainerUtil.addIfNotNull(result, getter);
    return result;
  }
}
