/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.javaee.web;

import com.intellij.openapi.util.text.StringUtil;
import org.jetbrains.annotations.NotNull;

/**
 * @author Dmitry Avdeev
*/
public enum ServletMappingType {

  /**
   * A string beginning with a '*.' prefix is used as an extension mapping.
   */
  EXTENSION {
      @Override
      public ServletMappingInfo createMappingInfo(@NotNull final String urlPattern, @NotNull final CommonServletMapping mapping) {
        return new ServletMappingInfo.ExtensionMappingInfo(urlPattern, mapping);
      }},

  /**
   * A string beginning with a '/' character and ending with a '/*' postfix is used for path mapping.
   */
  PATH {
      @Override
      public ServletMappingInfo createMappingInfo(@NotNull final String urlPattern, @NotNull final CommonServletMapping mapping) {
        return new ServletMappingInfo.PathMappingInfo(urlPattern, mapping);
      }},

  /**
   * A string containing only the '/' character indicates the "default" servlet of the
   * application. In this case the servlet path is the request URI minus the context
   * path and the path info is null.
   */
  DEFAULT {

    @Override
    public ServletMappingInfo createMappingInfo(@NotNull final String urlPattern, @NotNull final CommonServletMapping mapping) {
      return new ServletMappingInfo.DefaultMappingInfo(urlPattern, mapping);
    }},

  /**
   * All other strings are used for exact matches only.
   */
  EXACT {
      @Override
      public ServletMappingInfo createMappingInfo(@NotNull final String urlPattern, @NotNull final CommonServletMapping mapping) {
        return new ServletMappingInfo.ExactMappingInfo(urlPattern, mapping);
      }};

  public abstract ServletMappingInfo createMappingInfo(@NotNull String urlPattern, @NotNull CommonServletMapping mapping);

  @NotNull
  public static ServletMappingType getPatternType(@NotNull String pattern) {
    if (pattern.equals("/")) return DEFAULT;
    if (pattern.startsWith("*.")) return EXTENSION;
    if (StringUtil.startsWithChar(pattern, '/') && pattern.endsWith("/*")) return PATH;
    return EXACT;
  }

}
