package com.intellij.javaee.utils;

import com.intellij.jam.JavaLibraryUtils;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ModulePackageIndex;
import com.intellij.openapi.roots.PackageIndex;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiElement;
import com.intellij.psi.util.CachedValuesManager;
import org.jetbrains.annotations.NotNull;

import static com.intellij.psi.util.CachedValueProvider.Result.create;
import static com.intellij.psi.util.CachedValueProvider.Result.createSingleDependency;

public enum JavaeeType {
  JAKARTA("jakarta"), JAVAX("javax");

  public final String pkg;
  JavaeeType(String pkg) { this.pkg = pkg; }

  @NotNull
  public static JavaeeType discover(@NotNull Project project) {
    return CachedValuesManager.getManager(project).getCachedValue(project, () -> {
      if (DumbService.isDumb(project)) {
        return createSingleDependency(JAVAX, DumbService.getInstance(project).getModificationTracker());
      }
      return create(hasJakartaPackages(project) ? JAKARTA : JAVAX, ProjectRootManager.getInstance(project),
                    DumbService.getInstance(project).getModificationTracker());
    });
  }

  /**
   * @deprecated Consider using {@link JavaeeType#discover(Module, JavaeeClass)} instead.
   */
  @Deprecated
  public static @NotNull JavaeeType discover(@NotNull Module module) {
    return CachedValuesManager.getManager(module.getProject()).getCachedValue(module, () -> {
        final Project project = module.getProject();
        if (DumbService.isDumb(project)) {
          return createSingleDependency(JAVAX, DumbService.getInstance(project).getModificationTracker());
        } else {
          return create(hasJakartaPackages(module) ? JAKARTA : JAVAX, ProjectRootManager.getInstance(project),
                        DumbService.getInstance(project).getModificationTracker());
        }
      });
  }

  public static @NotNull JavaeeType discover(@NotNull Module module, @NotNull JavaeeClass hint) {
    return JavaLibraryUtils.hasLibraryClass(module, hint.jakarta()) ? JAKARTA : JAVAX;
  }

  /**
   * @deprecated Consider using {@link JavaeeType#discover(PsiElement, JavaeeClass)} instead.
   */
  @Deprecated
  public static @NotNull JavaeeType discover(@NotNull PsiElement context) {
    final Project project = context.getProject();
    if (!context.isValid() || project.isDisposed() || DumbService.isDumb(project)) return JAVAX;

    final Module module = ModuleUtilCore.findModuleForPsiElement(context);
    return module != null ? discover(module) : discover(project);
  }

  /**
   * Correctly discovers {@link JavaeeType} in modules with mixed jakarta/javax dependencies.
   */
  public static @NotNull JavaeeType discover(@NotNull PsiElement context, JavaeeClass hint) {
    Project project = context.getProject();
    if (!context.isValid() || project.isDisposed() || DumbService.isDumb(project)) return JAVAX;

    Module module = ModuleUtilCore.findModuleForPsiElement(context);
    if (module != null) {
      return discover(module, hint);
    }
    return JavaLibraryUtils.hasLibraryClass(project, hint.jakarta()) ? JAKARTA : JAVAX;
  }

  private static boolean hasJakartaPackages(@NotNull Project project) {
    final VirtualFile jakartaPackage = PackageIndex.getInstance(project).getDirsByPackageName(JAKARTA.pkg, true).findFirst();
    return jakartaPackage != null && isInLibrary(project, jakartaPackage);
  }

  private static boolean hasJakartaPackages(@NotNull Module module) {
    final VirtualFile jakartaPackage = ModulePackageIndex.getInstance(module).getDirsByPackageName(JAKARTA.pkg, true).findFirst();
    return  jakartaPackage != null && isInLibrary(module.getProject(), jakartaPackage);
  }

  private static boolean isInLibrary(@NotNull Project project, VirtualFile first) {
    return ProjectRootManager.getInstance(project).getFileIndex().isInLibrary(first);
  }
}
