/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.ui;

import com.intellij.icons.AllIcons;
import com.intellij.jam.JavaJamBundle;
import com.intellij.javaee.J2EEBundle;
import com.intellij.openapi.actionSystem.AnActionEvent;
import com.intellij.openapi.help.HelpManager;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.ui.Messages;
import com.intellij.ui.AnActionButton;
import com.intellij.ui.AnActionButtonRunnable;
import com.intellij.ui.DoubleClickListener;
import com.intellij.ui.ToolbarDecorator;
import com.intellij.ui.table.JBTable;
import com.intellij.util.xml.ui.CommittablePanel;
import org.jetbrains.annotations.NotNull;

import javax.swing.*;
import javax.swing.table.TableModel;
import java.awt.event.MouseEvent;

public abstract class TableWithCRUDButtons extends DisposableComponent implements CommittablePanel {
  protected JTable myTable;
  private final JPanel myPanel;
  protected final Project myProject;

  protected TableWithCRUDButtons(Project project, boolean showNewButton, boolean showEditButton) {
    myProject = project;
    myTable = new JBTable();
    myPanel = ToolbarDecorator.createDecorator(myTable)
      .setAddAction(showNewButton ? new AnActionButtonRunnable() {
        @Override
        public void run(AnActionButton button) {
          doNew();
        }
      } : null).setRemoveAction(new AnActionButtonRunnable() {
        @Override
        public void run(AnActionButton button) {
          if (isRemoveOk()) {
            doRemove();
          }
        }
      }).setEditAction(showEditButton ? new AnActionButtonRunnable() {
        @Override
        public void run(AnActionButton button) {
          doEdit();
        }
      } : null).addExtraAction(new AnActionButton(J2EEBundle.messagePointer("action.AnActionButton.text.help"), AllIcons.Actions.Help) {
        @Override
        public void actionPerformed(@NotNull AnActionEvent e) {
          HelpManager.getInstance().invokeHelp(getHelpID());
        }
      }).disableUpDownActions().createPanel();

    new DoubleClickListener() {
      @Override
      protected boolean onDoubleClick(@NotNull MouseEvent e) {
        doEdit();
        return true;
      }
    }.installOn(myTable);

    myTable.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
  }

  protected boolean isRemoveOk() {
    final Object o = getSelectedValue();
    String name = /*o instanceof NamedObject ? ((NamedObject)o).getName() : */String.valueOf(o);
    return Messages.showYesNoDialog(myProject, JavaJamBundle.message("confirmation.text.delete.element", name),
                                    J2EEBundle.message("action.name.delete"), null) == Messages.YES;
  }

  @Override
  public void commit() {
  }

  @Override
  public final void reset() {
    refreshModel();
  }

  protected abstract void refreshModelImpl();

  protected void refreshModel() {
    Object selectedValue = getSelectedValue();
    refreshModelImpl();
    setSelection(selectedValue);
  }

  protected void setSelection(Object selectedValue) {
    if (selectedValue != null) {
      int index = indexOf(selectedValue);
      if (index >= 0) {
        myTable.getSelectionModel().setSelectionInterval(index, index);
      }
    }
  }

  private int indexOf(Object selectedValue) {
    TableModel model = myTable.getModel();
    for (int i = 0; i < model.getRowCount(); i++) {
      if (model.getValueAt(i, 0) == selectedValue) {
        return i;
      }
    }
    return -1;
  }

  protected abstract void doEdit();

  protected abstract void doRemove();

  protected abstract void doNew();

  protected Object getSelectedValue() {
    final int selectedRow = myTable.getSelectedRow();
    if (selectedRow == -1) return null;
    return myTable.getModel().getValueAt(selectedRow, 0);
  }

  @Override
  public JComponent getComponent() {
    return myPanel;
  }

  protected abstract String getHelpID();
}
