/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.module.view.nodes;

import com.intellij.ide.DeleteProvider;
import com.intellij.ide.util.PsiNavigationSupport;
import com.intellij.ide.util.treeView.NodeDescriptor;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.model.common.CommonModelManager;
import com.intellij.jam.view.DefaultUserResponse;
import com.intellij.jam.view.JamDeleteProvider;
import com.intellij.openapi.actionSystem.CommonDataKeys;
import com.intellij.openapi.actionSystem.LangDataKeys;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.IndexNotReadyException;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.pom.Navigatable;
import com.intellij.psi.PsiFile;
import com.intellij.psi.xml.XmlFile;
import com.intellij.ui.SimpleTextAttributes;
import com.intellij.util.ReflectionUtil;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.ModelMergerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public abstract class JavaeeObjectDescriptor <P extends CommonModelElement> extends JavaeeNodeDescriptor<P> {

  protected JavaeeObjectDescriptor(P element, NodeDescriptor parentDescriptor, Object parameters) {
    super(parentDescriptor.getProject(), parentDescriptor, parameters, element);
  }

  protected JavaeeObjectDescriptor(P element, Project project, Object parameters) {
    super(project, null, parameters, element);
  }

  @Override
  public DeleteProvider getDeleteProvider() {
    final DefaultUserResponse response = new DefaultUserResponse(myProject);
    return new JamDeleteProvider(getJamElement(), response);
  }

  @Override
  public P updateElement() {
    if (!isValid()) {
      return null;
    }

    P element = getJamElement();
    try {
      if (!element.isValid()) return null;
    }
    catch (IndexNotReadyException e) {
      return null;
    }
    return getJamElement();
  }

  @Override
  public String getNewTooltip() {
    return null;
  }

  @Override
  public boolean isValid() {
    try {
      return getJamElement().isValid();
    }
    catch (IndexNotReadyException e) {
      return false;
    }
  }

  @Override
  protected void doUpdate() {
    super.doUpdate();
    final String textExt = getNewNodeTextExt();
    if (StringUtil.isNotEmpty(textExt)) {
      addColoredFragment(" (" + textExt + ")", SimpleTextAttributes.GRAY_ITALIC_ATTRIBUTES);
    }
  }

  @Nullable
  protected String getNewNodeTextExt() {
    return null;
  }

  @Override
  public Object getData(@NotNull final String dataId) {
    if (CommonDataKeys.NAVIGATABLE.getName().equals(dataId)) {
      final Object navigatable = getNavigatable();
      if (navigatable != null) {
        return navigatable;
      }
    }
    if (CommonDataKeys.PSI_ELEMENT.getName().equals(dataId)) {
      final P element = getJamElement();
      return element.isValid() ? element.getIdentifyingPsiElement() : null;
    }
    if (LangDataKeys.MODULE.getName().equals(dataId)) {
      return getParentElementOfType(Module.class);
    }
    return super.getData(dataId);
  }

  @Nullable
  public <T> T getParentElementOfType(final Class<T> elementClass) {
    for (NodeDescriptor cur = this; cur != null; cur = cur.getParentDescriptor()) {
      final Object element = cur.getElement();
      if (element != null && ReflectionUtil.isAssignable(elementClass, element.getClass())) {
        return (T)element;
      }
    }
    return null;
  }

  @Nullable
  protected Navigatable getNavigatable() {

    final DomElement domElement = CommonModelManager.getInstance().getDomElement(getJamElement());
    if (domElement != null && domElement.getXmlTag() != null) {
      final XmlFile file = DomUtil.getFile(domElement);
      final VirtualFile virtualFile = file.getVirtualFile();
      if (virtualFile != null && virtualFile.isValid()) {
        return PsiNavigationSupport.getInstance().createNavigatable(file.getProject(), virtualFile,
                                                                    domElement.getXmlTag().getTextRange()
                                                                              .getStartOffset());
      }
    }
    final CommonModelElement annoElement = ModelMergerUtil.getImplementation(getJamElement(), CommonModelElement.class);
    if (annoElement != null && annoElement.getIdentifyingPsiElement() != null) {
      final PsiFile file = annoElement.getContainingFile();
      final VirtualFile virtualFile = file == null? null : file.getVirtualFile();
      if (virtualFile != null && virtualFile.isValid()) {
        return PsiNavigationSupport.getInstance().createNavigatable(file.getProject(), virtualFile,
                                                                    annoElement.getIdentifyingPsiElement()
                                                                               .getNavigationElement().getTextRange()
                                                                               .getStartOffset());
      }
    }
    return null;
  }
}
