/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package com.intellij.javaee.model.jam;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamSimpleReferenceConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.*;
import com.intellij.javaee.utils.JavaeeType;
import com.intellij.javaee.web.CommonFilter;
import com.intellij.javaee.web.CommonServlet;
import com.intellij.javaee.web.CommonServletMapping;
import com.intellij.javaee.web.WebUtil;
import com.intellij.javaee.web.facet.WebFacet;
import com.intellij.psi.*;
import com.intellij.psi.ref.AnnotationChildLink;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.GenericValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;

import static com.intellij.javaee.web.WebCommonClassNames.ANNOTATION_WEB_FILTER;
import static java.util.Collections.emptyList;

/**
 * @author Dmitry Avdeev
 */
public abstract class JamFilter extends CommonModelElement.PsiBase implements CommonFilter, CommonServletMapping<CommonFilter>, JamElement {

  public static final JamStringAttributeMeta.Single<String> NAME_META = JamAttributeMeta.singleString("filterName");
  public static final JamStringAttributeMeta.Collection<String> URL_PATTERNS_META = JamAttributeMeta.collectionString("urlPatterns");
  public static final JamStringAttributeMeta.Collection<CommonServlet> SERVLET_NAMES_META =
    JamAttributeMeta.collectionString("servletNames", new JamSimpleReferenceConverter<>() {
      @Override
      public CommonServlet fromString(@Nullable final String s, JamStringAttributeElement<CommonServlet> context) {
        if (s == null) return null;

        PsiAnnotationMemberValue psiElement = context.getPsiElement();
        if (psiElement == null) return null;

        WebFacet facet = WebUtil.getWebFacet(psiElement);
        if (facet == null) return null;

        return ContainerUtil.find(facet.getWebModel().getServlets(),
                                  commonServlet -> s.equals(commonServlet.getServletName().getStringValue()));
      }

      @Override
      public Collection<CommonServlet> getVariants(JamStringAttributeElement<CommonServlet> context) {
        PsiAnnotationMemberValue psiElement = context.getPsiElement();
        if (psiElement == null) return null;

        WebFacet facet = WebUtil.getWebFacet(psiElement);
        if (facet == null) return emptyList();

        return facet.getWebModel().getServlets();
      }
    });

  public static final JamAnnotationAttributeMeta.Collection<JamInitParam> JAVAX_INIT_PARAMS_META =
    JamAttributeMeta.annoCollection("initParams", JamInitParam.JAVAX_INIT_PARAM_ANNO_META, JamInitParam.class);

  public static final JamAnnotationAttributeMeta.Collection<JamInitParam> JAKARTA_INIT_PARAMS_META =
    JamAttributeMeta.annoCollection("initParams", JamInitParam.JAKARTA_INIT_PARAM_ANNO_META, JamInitParam.class);

  private static final JamAnnotationArchetype FILTER_ARCHETYPE = new JamAnnotationArchetype().addAttribute(NAME_META).
    addAttribute(URL_PATTERNS_META).
    addAttribute(JAVAX_INIT_PARAMS_META).
    addAttribute(JAKARTA_INIT_PARAMS_META).
    addAttribute(SERVLET_NAMES_META);

  public static final JamAnnotationMeta JAVAX_FILTER_ANNO_META = new JamAnnotationMeta(ANNOTATION_WEB_FILTER.javax(), FILTER_ARCHETYPE);
  public static final JamAnnotationMeta JAKARTA_FILTER_ANNO_META = new JamAnnotationMeta(ANNOTATION_WEB_FILTER.jakarta(), FILTER_ARCHETYPE);

  public static final JamClassMeta<JamFilter> FILTER_CLASS_META = new JamClassMeta<>(JamFilter.class)
    .addAnnotation(JAVAX_FILTER_ANNO_META)
    .addAnnotation(JAKARTA_FILTER_ANNO_META);

  private final PsiClass myPsiClass;
  private final PsiElementRef<PsiAnnotation> myAnno;
  private final @NotNull JavaeeType myJavaeeType;

  @Override
  @NotNull
  public PsiClass getPsiClass() {
    return myPsiClass;
  }

  protected JamFilter(PsiClass aClass) {
    myPsiClass = aClass;
    myJavaeeType = ANNOTATION_WEB_FILTER.typeFromAnnotated(aClass);
    myAnno = AnnotationChildLink.createRef(aClass, ANNOTATION_WEB_FILTER.fqn(myJavaeeType));
  }

  @NotNull
  @Override
  public PsiElement getPsiElement() {
    return getPsiClass();
  }

  protected boolean isJavax() {
    return myJavaeeType.equals(JavaeeType.JAVAX);
  }

  @Override
  @NotNull
  public JamStringAttributeElement<String> getFilterName() {
    return isJavax() ?
           JAVAX_FILTER_ANNO_META.getAttribute(getPsiClass(), NAME_META) :
           JAKARTA_FILTER_ANNO_META.getAttribute(getPsiClass(), NAME_META);
  }

  @Override
  public List<JamStringAttributeElement<String>> getUrlPatterns() {
    return isJavax() ?
           JAVAX_FILTER_ANNO_META.getAttribute(getPsiClass(), URL_PATTERNS_META) :
           JAKARTA_FILTER_ANNO_META.getAttribute(getPsiClass(), URL_PATTERNS_META);
  }

  @Override
  public List<? extends GenericValue<CommonServlet>> getServletNames() {
    PsiElementRef<PsiAnnotation> annotationRef = isJavax() ?
                                                 JAVAX_FILTER_ANNO_META.getAnnotationRef(getPsiClass()) :
                                                 JAKARTA_FILTER_ANNO_META.getAnnotationRef(getPsiClass());
    return SERVLET_NAMES_META.getJam(annotationRef);
  }

  @Override
  public List<JamInitParam> getInitParams() {
    return isJavax() ? JAVAX_INIT_PARAMS_META.getJam(myAnno) : JAKARTA_INIT_PARAMS_META.getJam(myAnno);
  }

  @Override
  public JamInitParam addInitParam() {
    return isJavax() ?
           JAVAX_INIT_PARAMS_META.addAttribute(PsiElementRef.real(JAVAX_FILTER_ANNO_META.getAnnotation(getPsiClass()))) :
           JAKARTA_INIT_PARAMS_META.addAttribute(PsiElementRef.real(JAKARTA_FILTER_ANNO_META.getAnnotation(getPsiClass())));
  }

  @Override
  public CommonFilter getServlet() {
    return this;
  }

  @Override
  public PsiElement getMappingElement() {
    return JAVAX_FILTER_ANNO_META.getAnnotation(getPsiClass());
  }
}
