// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.run.lifecycle.tabs;

import com.intellij.execution.process.ProcessHandler;
import com.intellij.openapi.Disposable;
import com.intellij.openapi.actionSystem.AnAction;
import com.intellij.openapi.actionSystem.AnActionEvent;
import com.intellij.openapi.actionSystem.DataProvider;
import com.intellij.openapi.actionSystem.ToggleAction;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NlsActions;
import com.intellij.openapi.util.NlsContexts;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.spring.boot.SpringBootApiBundle;
import com.intellij.spring.boot.library.SpringBootLibraryUtil;
import com.intellij.spring.boot.run.SpringBootApplicationRunConfigurationBase;
import com.intellij.spring.boot.run.lifecycle.*;
import com.intellij.ui.AppUIUtil;
import com.intellij.ui.JBColor;
import com.intellij.ui.ScrollPaneFactory;
import com.intellij.ui.components.JBLabel;
import com.intellij.ui.components.JBLoadingPanel;
import com.intellij.util.text.DateFormatUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.awt.*;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author konstantin.aleev
 */
public abstract class EndpointTab<T> implements Disposable, DataProvider {
  private static final String MESSAGE_CARD = "message";
  private static final String ENDPOINT_CARD = "endpoint";

  private final Endpoint<T> myEndpoint;
  private final SpringBootApplicationRunConfigurationBase myRunConfiguration;
  private final ProcessHandler myProcessHandler;
  private final LiveProperty.LivePropertyListener myEndpointListener;

  private final CardLayout myRootPanelLayout;
  private final JPanel myRootPanel;
  private final JBLoadingPanel myMessagePanel;
  private final JLabel myMessageLabel;

  private TooltipChangeListener myTooltipChangeListener;

  private volatile boolean myActuatorsEnabled = false;

  protected EndpointTab(@NotNull Endpoint<T> endpoint, @NotNull SpringBootApplicationRunConfigurationBase runConfiguration,
                        @NotNull ProcessHandler processHandler) {
    myEndpoint = endpoint;
    myRunConfiguration = runConfiguration;
    myProcessHandler = processHandler;

    myEndpointListener = new LiveProperty.LivePropertyListener() {
      @Override
      public void propertyChanged() {
        AppUIUtil.invokeLaterIfProjectAlive(getProject(), () -> {
          if (myMessagePanel.isLoading()) {
            myMessagePanel.stopLoading();
          }
          myRootPanelLayout.show(myRootPanel, ENDPOINT_CARD);
        });
        updateComponent();
      }

      @Override
      public void computationFailed(@NotNull Exception e) {
        StringBuilder messageBuilder = new StringBuilder();
        if (!(e instanceof SpringBootApplicationConfigurationException)) {
          messageBuilder.append(e.getClass().getName()).append(": ");
        }
        messageBuilder.append(e.getLocalizedMessage());

        Set<Throwable> causes = new HashSet<>();
        Throwable parent = e;
        Throwable cause = e.getCause();
        causes.add(parent);
        while (cause != null && !causes.contains(cause)) {
          messageBuilder.append("<br>").append(SpringBootApiBundle.message("spring.boot.application.endpoints.error.caused.by",
                                                                           cause.getClass().getName(),
                                                                           cause.getLocalizedMessage()));
          parent = cause;
          cause = parent.getCause();
          causes.add(parent);
        }
        showMessage(getErrorMessage(messageBuilder.toString()));
      }
    };

    myRootPanelLayout = new CardLayout();
    myRootPanel = new JPanel(myRootPanelLayout);

    myMessagePanel = new JBLoadingPanel(new GridBagLayout(), this);
    final String name = StringUtil.shortenTextWithEllipsis(myRunConfiguration.getName(), 30, 3);
    myMessagePanel.setLoadingText(SpringBootApiBundle.message("spring.boot.application.endpoints.application.is.starting", name));
    myMessagePanel.startLoading();

    myMessageLabel = new JBLabel();
    myMessageLabel.setForeground(JBColor.GRAY);

    GridBagConstraints gbc = new GridBagConstraints();
    gbc.anchor = GridBagConstraints.CENTER;
    gbc.fill = GridBagConstraints.BOTH;
    gbc.weighty = 0.66;
    myMessagePanel.add(myMessageLabel, gbc);

    // Add bottom spacer
    gbc.weighty = 0.33;
    gbc.gridy = 1;
    myMessagePanel.add(new JBLabel(), gbc);

    myRootPanel.add(MESSAGE_CARD, ScrollPaneFactory.createScrollPane(myMessagePanel,
                                                                     ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
                                                                     ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED));
  }

  @NotNull
  public String getId() {
    return myEndpoint.getId();
  }

  public abstract @NlsContexts.TabTitle String getTitle();

  @NotNull
  public abstract Icon getIcon();

  @Override
  public void dispose() {
    SpringBootApplicationInfo info = getInfo();
    if (info != null) {
      getLiveProperty(info).removePropertyListener(myEndpointListener);
    }
    myTooltipChangeListener = null;
  }

  @Nullable
  @Override
  public Object getData(@NotNull String dataId) {
    return null;
  }

  public final JComponent getComponent() {
    myRootPanel.add(ENDPOINT_CARD, getEndpointComponent());
    return myRootPanel;
  }

  public final void refresh() {
    SpringBootApplicationInfo info = getInfo();
    if (info != null) {
      getLiveProperty(info).compute();
    }
  }

  public final void initPropertyListeners(@NotNull SpringBootApplicationInfo info) {
    getLiveProperty(info).addPropertyListener(myEndpointListener);
    doInitPropertyListeners(info);
  }

  public final void showMessage(@Nullable @NlsContexts.Label String message) {
    AppUIUtil.invokeLaterIfProjectAlive(getProject(), () -> {
      if (myMessagePanel.isLoading()) {
        myMessagePanel.stopLoading();
      }
      myRootPanelLayout.show(myRootPanel, MESSAGE_CARD);
      myMessageLabel.setText(message);
    });
  }

  public final void showLoading() {
    AppUIUtil.invokeLaterIfProjectAlive(getProject(), () -> {
      myMessageLabel.setText(null);
      myMessagePanel.startLoading();
      myRootPanelLayout.show(myRootPanel, MESSAGE_CARD);
    });
  }

  public final void setTooltipChangeListener(@Nullable TooltipChangeListener tooltipChangeListener) {
    myTooltipChangeListener = tooltipChangeListener;
  }

  @NotNull
  protected final Project getProject() {
    return myRunConfiguration.getProject();
  }

  protected final boolean isActuatorsEnabled() {
    return myActuatorsEnabled;
  }

  public void checkAvailability() {
    myActuatorsEnabled = SpringBootLibraryUtil.hasActuators(getRunConfiguration().getModule());
    if (!myActuatorsEnabled) {
      showMessage(SpringBootApiBundle.message("spring.boot.application.endpoints.error.actuator.starter.disabled"));
    }
  }

  public void updateRefreshAction(AnActionEvent e, @NotNull SpringBootApplicationInfo info) {
    e.getPresentation().setEnabled(myActuatorsEnabled);
  }

  @NotNull
  protected final SpringBootApplicationRunConfigurationBase getRunConfiguration() {
    return myRunConfiguration;
  }

  @NotNull
  protected final ProcessHandler getProcessHandler() {
    return myProcessHandler;
  }

  @NotNull
  protected final JPanel getRootPanel() {
    return myRootPanel;
  }

  private void setTimeStampTooltip(long timeStamp) {
    if (myTooltipChangeListener != null) {
      String message = null;
      if (timeStamp > 0) {
        message =
          SpringBootApiBundle.message("spring.boot.application.endpoints.updated.at", DateFormatUtil.formatTimeWithSeconds(timeStamp));
      }
      myTooltipChangeListener.tooltipChanged(message);
    }
  }

  protected final boolean isTabLoading() {
    return myMessagePanel.isLoading();
  }

  @Nullable
  protected final SpringBootApplicationInfo getInfo() {
    Project project = getProject();
    if (!project.isOpen() || project.isDisposed()) {
      return null;
    }
    return SpringBootApplicationLifecycleManager.getInstance(project).getSpringBootApplicationInfo(myProcessHandler);
  }

  protected final void infoRemoved() {
    AppUIUtil.invokeLaterIfProjectAlive(getProject(), () -> {
      if (myMessagePanel.isLoading()) {
        myMessagePanel.stopLoading();
      }
    });
  }

  protected final void updateActionPresentation(@NotNull AnActionEvent e) {
    SpringBootApplicationInfo info = getInfo();
    e.getPresentation().setEnabled(info != null && Boolean.TRUE.equals(info.getReadyState().getValue()) && !isTabLoading());
  }

  @NotNull
  protected List<AnAction> getToolbarActions() {
    return Collections.emptyList();
  }

  protected void doInitPropertyListeners(@NotNull SpringBootApplicationInfo info) {
  }

  protected @NlsContexts.Label String getErrorMessage(String cause) {
    return SpringBootApiBundle.message("spring.boot.application.endpoints.error.failed.to.retrieve.endpoint.data.detailed",
                                       myEndpoint.getId(),
                                       cause);
  }

  @NotNull
  protected LiveProperty<T> getLiveProperty(SpringBootApplicationInfo info) {
    return info.getEndpointData(myEndpoint);
  }

  @NotNull
  protected abstract JComponent getEndpointComponent();

  private void updateComponent() {
    SpringBootApplicationInfo info = SpringBootApplicationLifecycleManager.getInstance(getProject())
                                                                          .getSpringBootApplicationInfo(getProcessHandler());
    LiveProperty<T> liveProperty = info != null ? getLiveProperty(info) : null;
    final T value = liveProperty != null ? liveProperty.getValue() : null;
    final long timeStamp = liveProperty != null ? liveProperty.getTimeStamp() : -1L;
    AppUIUtil.invokeLaterIfProjectAlive(getProject(), () -> {
      setTimeStampTooltip(timeStamp);
      doUpdateComponent(value);
    });
  }

  protected abstract void doUpdateComponent(@Nullable T value);

  interface TooltipChangeListener {
    void tooltipChanged(@Nullable @NlsContexts.Tooltip String tooltip);
  }

  protected abstract class EndpointToggleAction extends ToggleAction {
    protected EndpointToggleAction(@Nullable @NlsActions.ActionText String text) {
      super(text);
    }

    protected EndpointToggleAction(@Nullable @NlsActions.ActionText String text,
                                   @Nullable @NlsActions.ActionDescription String description,
                                   @Nullable final Icon icon) {
      super(text, description, icon);
    }

    @Override
    public void update(@NotNull AnActionEvent e) {
      super.update(e);
      updateActionPresentation(e);
    }
  }
}
