// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.run.lifecycle;

import com.intellij.execution.process.ProcessHandler;
import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Key;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * @author konstantin.aleev
 */
public interface SpringBootApplicationLifecycleManager {
  Key<Integer> JMX_PORT = Key.create("SPRING_BOOT_APPLICATION_JMX_PORT");

  static SpringBootApplicationLifecycleManager getInstance(Project project) {
    return ServiceManager.getService(project, SpringBootApplicationLifecycleManager.class);
  }

  boolean isLifecycleManagementEnabled(@NotNull ProcessHandler handler);

  /**
   * @param handler process handler associated with Spring Boot application.
   * @return {@code null} if lifecycle support is not enabled for the given process handler
   * or associated process is not started or already stopped.
   */
  @Nullable
  SpringBootApplicationInfo getSpringBootApplicationInfo(ProcessHandler handler);

  List<SpringBootApplicationInfo> getSpringBootApplicationInfos();

  /**
   * @param handler process handler.
   * @return {@code null} if process handler is not associated with Spring Boot application
   * or associated process is not started or already stopped.
   */
  @Nullable
  SpringBootApplicationDescriptor getSpringBootApplicationDescriptor(ProcessHandler handler);

  List<SpringBootApplicationDescriptor> getRunningSpringBootApplications();

  void addInfoListener(@NotNull InfoListener listener);

  void removeInfoListener(@NotNull InfoListener listener);

  interface InfoListener {
    void infoAdded(@NotNull ProcessHandler processHandler, @NotNull SpringBootApplicationInfo info);

    void infoRemoved(@NotNull ProcessHandler processHandler, @NotNull SpringBootApplicationInfo info);
  }

  /**
   * For application running on targets, represents the mapping from remote port the application actually runs on to
   * its local tunnel counterpart. For application running locally, this mapping is an identity.
   */
  interface TunnelPortMapping {
    int getLocalPort(int applicationPort);

    Key<TunnelPortMapping> KEY = Key.create("SPRING_BOOT_APPLICATION_TUNNEL_PORT_MAPPING");
  }
}
