// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model.autoconfigure.conditions;

import com.intellij.microservices.config.MetaConfigKey;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.UserDataHolderBase;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiManager;
import com.intellij.spring.boot.model.ConfigurationValueResult;
import com.intellij.spring.boot.model.ConfigurationValueSearchParams;
import com.intellij.spring.boot.model.SpringBootModelConfigFileContributor;
import com.intellij.util.PairProcessor;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Map;
import java.util.Set;

public class ConditionalOnEvaluationContextBase extends UserDataHolderBase implements ConditionalOnEvaluationContext {

  @NotNull
  private final PsiClass myAutoConfigClass;

  @NotNull
  private final Module myModule;

  @Nullable
  private final Set<String> myActiveProfiles;

  private final Map<SpringBootModelConfigFileContributor, List<VirtualFile>> myContributors;

  public ConditionalOnEvaluationContextBase(@NotNull PsiClass autoConfigClass,
                                            @NotNull Module module,
                                            @Nullable Set<String> activeProfiles,
                                            @NotNull Map<SpringBootModelConfigFileContributor, List<VirtualFile>> contributors,
                                            @Nullable ConditionalOnEvaluationContextBase sharedContext) {
    myAutoConfigClass = autoConfigClass;
    myModule = module;
    myActiveProfiles = activeProfiles;
    myContributors = contributors;

    if (sharedContext != null) {
      sharedContext.copyUserDataTo(this);
    }
  }

  @NotNull
  @Override
  public PsiClass getAutoConfigClass() {
    return myAutoConfigClass;
  }

  @NotNull
  @Override
  public Module getModule() {
    return myModule;
  }

  @Nullable
  @Override
  public Set<String> getActiveProfiles() {
    return myActiveProfiles;
  }

  @Override
  public boolean processConfigurationFiles(PairProcessor<SpringBootModelConfigFileContributor, List<VirtualFile>> processor) {
    for (SpringBootModelConfigFileContributor contributor : SpringBootModelConfigFileContributor.EP_NAME.getExtensions()) {
      if (!processor.process(contributor, myContributors.get(contributor))) return false;
    }
    return true;
  }

  @Override
  public boolean processConfigurationValues(Processor<List<ConfigurationValueResult>> processor,
                                            boolean checkRelaxedNames,
                                            MetaConfigKey @NotNull ... configKeys) {
    final PsiManager psiManager = PsiManager.getInstance(getModule().getProject());
    return processConfigurationFiles((contributor, files) -> {
      for (VirtualFile file : files) {
        final PsiFile psiFile = psiManager.findFile(file);
        if (psiFile == null) continue;

        for (MetaConfigKey key : configKeys) {
          ConfigurationValueSearchParams params =
            new ConfigurationValueSearchParams(getModule(), psiFile, checkRelaxedNames, getActiveProfiles(), key);
          final List<ConfigurationValueResult> values = contributor.findConfigurationValues(params);
          if (!processor.process(values)) return false;
        }

        if (ContainerUtil.isEmpty(getActiveProfiles())) break;
      }
      return true;
    });
  }
}
