// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model;

import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.fileTypes.FileType;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.roots.ModuleRootManager;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.openapi.vfs.VirtualFileManager;
import com.intellij.psi.PsiClass;
import com.intellij.psi.search.FileTypeIndex;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.GlobalSearchScopesCore;
import com.intellij.spring.boot.application.SpringBootApplicationService;
import com.intellij.spring.facet.SpringFacet;
import com.intellij.spring.facet.SpringFileSet;
import com.intellij.spring.facet.beans.CustomSetting;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;
import org.jetbrains.jps.model.java.JavaResourceRootType;

import java.util.*;

/**
 * Collects all default ({@code application|bootstrap}) and/or custom named config files located in resource roots for the given filetype.
 */
public abstract class SpringBootModelConfigFileContributor {
  public static final ExtensionPointName<SpringBootModelConfigFileContributor> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.boot.modelConfigFileContributor");

  private static final Comparator<VirtualFile> VF_BY_NAME_COMPARATOR = (o1, o2) -> StringUtil.naturalCompare(o1.getName(), o2.getName());

  private static final boolean OUR_TEST_MODE = ApplicationManager.getApplication().isUnitTestMode();

  private final FileType myFileType;

  protected SpringBootModelConfigFileContributor(FileType fileType) {
    myFileType = fileType;
  }

  public FileType getFileType() {
    return myFileType;
  }

  /**
   * Returns all default and user configuration files.
   *
   * @param module           Module to evaluate.
   * @param includeTestScope
   * @return Configuration files for this contributor.
   */
  public List<VirtualFile> getConfigurationFiles(Module module, boolean includeTestScope) {
    List<VirtualFile> files = new SmartList<>();
    for (SpringBootModelConfigFileNameContributor fileNameContributor : SpringBootModelConfigFileNameContributor.EP_NAME.getExtensions()) {
      if (fileNameContributor.accept(module)) {
        files.addAll(getConfigurationFiles(module, fileNameContributor, includeTestScope));
      }
    }
    return files;
  }

  /**
   * Returns all default and user configuration files applicable for the given file set.
   *
   * @param module           Module to evaluate.
   * @param fileSet          Fileset to evaluate.
   * @param includeTestScope
   * @return Configuration files for this contributor.
   */
  public List<VirtualFile> getConfigurationFiles(Module module, SpringFileSet fileSet, boolean includeTestScope) {
    List<VirtualFile> files = new SmartList<>();
    for (SpringBootModelConfigFileNameContributor fileNameContributor : SpringBootModelConfigFileNameContributor.EP_NAME.getExtensions()) {
      if (fileNameContributor.accept(fileSet)) {
        files.addAll(getConfigurationFiles(module, fileNameContributor, includeTestScope));
        break;
      }
    }
    return files;
  }

  public List<VirtualFile> getConfigurationFiles(Module module, SpringBootModelConfigFileNameContributor fileNameContributor,
                                                 boolean includeTestScope) {
    final String springConfigName = getSpringConfigName(module, fileNameContributor);
    final List<VirtualFile> userConfigurationFiles = findApplicationConfigFiles(module, includeTestScope, springConfigName);
    return ContainerUtil.concat(userConfigurationFiles, findCustomConfigFiles(module, fileNameContributor));
  }

  private static List<VirtualFile> findCustomConfigFiles(Module module, SpringBootModelConfigFileNameContributor fileNameContributor) {
    final SpringFacet springFacet = getRelevantFacet(module);
    if (springFacet == null) { // adding new facet -> edit dialog
      return Collections.emptyList();
    }

    final CustomSetting.STRING setting = springFacet.findSetting(fileNameContributor.getCustomFilesSettingDescriptor().key);
    assert setting != null;
    final String value = setting.getStringValue();
    if (value == null) {
      return Collections.emptyList();
    }

    final List<String> urls = StringUtil.split(value, ";");
    List<VirtualFile> files = new ArrayList<>(urls.size());
    for (String url : urls) {
      final VirtualFile configFile = VirtualFileManager.getInstance().findFileByUrl(url);
      ContainerUtil.addIfNotNull(files, configFile);
    }
    return files;
  }

  @NotNull
  private static String getSpringConfigName(Module module, SpringBootModelConfigFileNameContributor fileNameContributor) {
    final SpringFacet springFacet = getRelevantFacet(module);
    if (springFacet == null) { // adding new facet -> edit dialog
      return fileNameContributor.getCustomNameSettingDescriptor().defaultValue;
    }

    final CustomSetting.STRING setting = springFacet.findSetting(fileNameContributor.getCustomNameSettingDescriptor().key);
    assert setting != null;

    return StringUtil.notNullize(setting.getStringValue(), setting.getDefaultValue());
  }

  @Nullable
  private static SpringFacet getRelevantFacet(Module module) {
    final SpringFacet facet = findSpringBootApplicationFacet(module);
    if (facet != null) {
      return facet;
    }

    if (module.isDisposed()) return null;

    // Gradle: test-only module has dependency to main module with SB main class + possible custom configuration
    for (Module dependentModule : ModuleRootManager.getInstance(module).getDependencies()) {
      final SpringFacet dependentFacet = findSpringBootApplicationFacet(dependentModule);
      if (dependentFacet != null) {
        return dependentFacet;
      }
    }

    return null;
  }

  @Nullable
  private static SpringFacet findSpringBootApplicationFacet(Module module) {
    final SpringFacet springFacet = SpringFacet.getInstance(module);
    if (springFacet == null) {
      return null;
    }

    final List<PsiClass> applications = SpringBootApplicationService.getInstance().getSpringApplications(module);
    return applications.isEmpty() ? null : springFacet;
  }

  public List<VirtualFile> findApplicationConfigFiles(Module module,
                                                      boolean includeTestScope,
                                                      @NotNull String baseName) {
    final List<VirtualFile> productionConfigFiles = findConfigFilesInScope(module, false, baseName);
    if (!includeTestScope) {
      return productionConfigFiles;
    }

    final List<VirtualFile> testConfigFiles = findConfigFilesInScope(module, true, baseName);
    return ContainerUtil.concat(productionConfigFiles, testConfigFiles);
  }

  private List<VirtualFile> findConfigFilesInScope(Module module,
                                                   boolean testScope,
                                                   @NotNull String baseName) {
    final GlobalSearchScope configFileSearchScope = getConfigFileSearchScope(module, testScope);
    if (configFileSearchScope == null) {
      return Collections.emptyList();
    }

    final String fileNamePrefix = baseName + '-';

    final Collection<VirtualFile> baseNameConfigFiles = new SmartList<>();
    final Collection<VirtualFile> profileConfigFiles = new SmartList<>();
    FileTypeIndex.processFiles(myFileType, file -> {
      final String fileName = file.getNameWithoutExtension();
      if (fileName.equals(baseName)) {
        baseNameConfigFiles.add(file);
      }
      else if (StringUtil.startsWith(fileName, fileNamePrefix)) {
        profileConfigFiles.add(file);
      }
      return true;
    }, configFileSearchScope);

    // no matching files at all
    if (baseNameConfigFiles.isEmpty() &&
        profileConfigFiles.isEmpty()) {
      return Collections.emptyList();
    }

    // 0 -> find profile configs, 1 -> find related profile configs in same dir, >1 -> no results
    if (baseNameConfigFiles.size() > 1) {
      return Collections.emptyList();
    }

    @Nullable final VirtualFile baseNameConfigFile = ContainerUtil.getFirstItem(baseNameConfigFiles);

    List<VirtualFile> result = new SmartList<>();
    ContainerUtil.addIfNotNull(result, baseNameConfigFile);

    if (profileConfigFiles.isEmpty()) {
      return result;
    }

    final List<VirtualFile> matchingProfileConfigFiles;
    if (baseNameConfigFile != null) {
      final VirtualFile parentDirectory = baseNameConfigFile.getParent();
      if (parentDirectory == null) return result;

      GlobalSearchScope containingDirectoryScope = GlobalSearchScopesCore.directoryScope(module.getProject(), parentDirectory, false);
      matchingProfileConfigFiles = ContainerUtil.filter(profileConfigFiles, file -> containingDirectoryScope.contains(file));
    }
    else {
      matchingProfileConfigFiles = new ArrayList<>(profileConfigFiles);
    }

    matchingProfileConfigFiles.sort(VF_BY_NAME_COMPARATOR);
    result.addAll(matchingProfileConfigFiles);

    return result;
  }

  @Nullable
  public GlobalSearchScope getConfigFileSearchScope(Module module, boolean testScope) {
    if (OUR_TEST_MODE && !USE_RESOURCE_ROOTS_FOR_TESTS) {
      return module.getModuleScope(testScope);
    }

    if (module.isDisposed()) {
      return null;
    }

    final ModuleRootManager moduleRootManager = ModuleRootManager.getInstance(module);
    final List<VirtualFile> resourceRoots =
      moduleRootManager.getSourceRoots(testScope ? JavaResourceRootType.TEST_RESOURCE : JavaResourceRootType.RESOURCE);
    if (resourceRoots.isEmpty()) {
      return null;
    }

    return GlobalSearchScopesCore.directoriesScope(module.getProject(), true, resourceRoots.toArray(VirtualFile.EMPTY_ARRAY));
  }

  /**
   * Returns actual configuration value(s) if found in file for given profiles.
   * <p/>
   * Use {@link SpringBootConfigValueSearcher} for direct value search/processing.
   *
   * @param params Search parameters.
   * @return Actual configuration value(s) or empty list if no occurrence(s) of key.
   */
  @NotNull
  public abstract List<ConfigurationValueResult> findConfigurationValues(ConfigurationValueSearchParams params);

  protected static boolean isProfileRelevant(@NotNull ConfigurationValueSearchParams params, @Nullable String fileNameSuffix) {
    Set<String> activeProfiles = params.getActiveProfiles();
    return fileNameSuffix == null ||
           ContainerUtil.isEmpty(activeProfiles) ||
           params.isProcessAllProfiles() ||
           activeProfiles.contains(fileNameSuffix);
  }

  private static boolean USE_RESOURCE_ROOTS_FOR_TESTS = false;

  @TestOnly
  public static void setUseResourceRootsForTests(boolean useResourceRootsForTests) {
    USE_RESOURCE_ROOTS_FOR_TESTS = useResourceRootsForTests;
  }
}
