// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model

import com.intellij.openapi.module.Module
import com.intellij.openapi.util.Ref
import com.intellij.psi.PsiManager
import com.intellij.microservices.config.MetaConfigKey.Deprecation
import com.intellij.spring.boot.application.metadata.SpringBootApplicationMetaConfigKeyManager
import com.intellij.util.Processor
import com.intellij.util.containers.ContainerUtil

/**
 * Search/process configuration files values.
 */
class SpringBootConfigValueSearcher(private val myModule: Module, private val myIncludeTests: Boolean = false,
                                    private val myConfigKey: String, private val myCheckRelaxedNames: Boolean = true,
                                    private val myActiveProfiles: Set<String>? = null,
                                    private val myKeyIndex: String? = null, private val myKeyProperty: String? = null) {

  fun findValueText(): String? {
    val valueText = Ref.create<String>()
    val findValueTextProcessor = Processor<ConfigurationValueResult> { result ->
      val text = result.valueText ?: return@Processor true

      valueText.set(text)
      false
    }
    process(findValueTextProcessor)
    return valueText.get()
  }

  fun process(processor: Processor<ConfigurationValueResult>): Boolean {
    val metaConfigKey = SpringBootApplicationMetaConfigKeyManager.getInstance().findApplicationMetaConfigKey(myModule,
                                                                                                             myConfigKey) ?: return true
    // keys with ERROR level are not used by SB, skip them
    if (metaConfigKey.deprecation.level == Deprecation.DeprecationLevel.ERROR) {
      return true
    }

    val psiManager = PsiManager.getInstance(myModule.project)
    for (contributor in SpringBootModelConfigFileContributor.EP_NAME.extensions) {
      for (virtualFile in contributor.getConfigurationFiles(myModule, myIncludeTests)) {
        val configPsiFile = psiManager.findFile(virtualFile) ?: continue

        val params = ConfigurationValueSearchParams(myModule, configPsiFile, myCheckRelaxedNames, myActiveProfiles, metaConfigKey,
                                                    myKeyIndex, myKeyProperty)
        val result = contributor.findConfigurationValues(params)
        if (!ContainerUtil.process(result, processor)) return false
      }
    }

    return true
  }

  companion object {

    @JvmStatic
    fun productionForProfiles(module: Module, configKey: String, activeProfiles: Set<String>?): SpringBootConfigValueSearcher {
      return SpringBootConfigValueSearcher(myModule = module, myConfigKey = configKey, myActiveProfiles = activeProfiles)
    }

    @JvmStatic
    fun productionForProfiles(module: Module,
                              configKey: String,
                              activeProfiles: Set<String>?,
                              keyIndex: String?,
                              keyProperty: String?): SpringBootConfigValueSearcher {
      return SpringBootConfigValueSearcher(myModule = module, myConfigKey = configKey, myActiveProfiles = activeProfiles,
                                           myKeyIndex = keyIndex, myKeyProperty = keyProperty)
    }
  }
}
