// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata.additional;

import com.intellij.codeInsight.daemon.DaemonCodeAnalyzer;
import com.intellij.codeInsight.intention.HighPriorityAction;
import com.intellij.codeInspection.LocalQuickFixAndIntentionActionOnPsiElement;
import com.intellij.icons.AllIcons;
import com.intellij.json.psi.*;
import com.intellij.openapi.application.WriteAction;
import com.intellij.openapi.command.WriteCommandAction;
import com.intellij.openapi.editor.Editor;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.project.ProjectUtil;
import com.intellij.openapi.ui.Messages;
import com.intellij.openapi.ui.popup.JBPopupFactory;
import com.intellij.openapi.util.Iconable;
import com.intellij.openapi.util.ThrowableComputable;
import com.intellij.openapi.vfs.ReadonlyStatusHandler;
import com.intellij.openapi.vfs.VfsUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiManager;
import com.intellij.psi.codeStyle.CodeStyleManager;
import com.intellij.spring.boot.SpringBootApiBundle;
import com.intellij.spring.boot.SpringBootConfigFileConstants;
import com.intellij.spring.boot.application.metadata.SpringBootMetadataConstants;
import com.intellij.ui.SimpleListCellRenderer;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.ObjectUtils;
import com.intellij.util.Processor;
import com.intellij.util.ThrowableRunnable;
import icons.SpringBootApiIcons;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.io.IOException;
import java.util.List;

public class DefineLocalMetaConfigKeyFix extends LocalQuickFixAndIntentionActionOnPsiElement implements HighPriorityAction, Iconable {

  private final String myKeyName;

  public DefineLocalMetaConfigKeyFix(PsiElement psiElement, String keyName) {
    super(psiElement);
    myKeyName = keyName;
  }

  @Override
  public Icon getIcon(int flags) {
    return SpringBootApiIcons.SpringBoot;
  }

  @NotNull
  @Override
  public String getText() {
    return SpringBootApiBundle.message("DefineLocalMetaConfigKeyFix",  myKeyName);
  }

  @NotNull
  @Override
  public String getFamilyName() {
    return SpringBootApiBundle.message("DefineLocalMetaConfigKeyFix.define.configuration.key");
  }

  @Override
  public boolean startInWriteAction() {
    return false;
  }

  @Override
  public void invoke(@NotNull Project project,
                     @NotNull PsiFile file,
                     @Nullable Editor editor,
                     @NotNull PsiElement startElement,
                     @NotNull PsiElement endElement) {
    Module module = ModuleUtilCore.findModuleForPsiElement(startElement);
    if (module == null) {
      return;
    }
    String moduleName = module.getName();

    SpringBootAdditionalConfigUtils additionalConfigUtils = new SpringBootAdditionalConfigUtils(module);
    if (!additionalConfigUtils.hasResourceRoots()) {
      Messages.showWarningDialog(project,
                                 SpringBootApiBundle.message("spring.boot.properties.no.resources.roots", moduleName),
                                 SpringBootApiBundle.message("spring.boot.name"));
      return;
    }

    Processor<JsonFile> addKeyToExistingProcessor = jsonFile -> {
      addKey(project, jsonFile);
      return false;
    };
    if (!additionalConfigUtils.processAdditionalMetadataFiles(addKeyToExistingProcessor)) {
      return;
    }

    List<VirtualFile> roots = additionalConfigUtils.getResourceRoots();
    if (editor == null) {
      createMetadataJson(project, roots.get(0));
    }
    else {
      SimpleListCellRenderer<VirtualFile> renderer = SimpleListCellRenderer.create((label, value, index) -> {
        label.setText(ProjectUtil.calcRelativeToProjectPath(value, project));
        label.setIcon(AllIcons.Modules.ResourcesRoot);
      });

      JBPopupFactory.getInstance()
        .createPopupChooserBuilder(roots)
        .setTitle(SpringBootApiBundle.message("spring.boot.properties.no.json.metadata.popup", myKeyName))
        .setAdText(SpringBootApiBundle.message("spring.boot.properties.no.json.metadata.hint",
                                               SpringBootConfigFileConstants.ADDITIONAL_SPRING_CONFIGURATION_METADATA_JSON))
        .setRenderer(renderer)
        .setItemChosenCallback(selectedRoot -> {
          if (selectedRoot == null) return;

          createMetadataJson(project, selectedRoot);

          DaemonCodeAnalyzer.getInstance(project).restart(file);
        })
        .setRequestFocus(true)
        .createPopup()
        .showInBestPositionFor(editor);
    }
  }

  private void createMetadataJson(@NotNull Project project, @NotNull VirtualFile selectedRoot) {
    WriteCommandAction.writeCommandAction(project)
      .withName(SpringBootApiBundle.message("spring.boot.properties.json.metadata.create", myKeyName))
      .run(() -> {
        try {
          VirtualFile metaInf = VfsUtil.createDirectoryIfMissing(selectedRoot, "META-INF");

          VirtualFile addVf =
            metaInf.createChildData(project, SpringBootConfigFileConstants.ADDITIONAL_SPRING_CONFIGURATION_METADATA_JSON);

          VfsUtil.saveText(addVf, "{ \"" + SpringBootMetadataConstants.PROPERTIES + "\": [ ] }");

          JsonFile jsonFile = (JsonFile)PsiManager.getInstance(project).findFile(addVf);
          assert jsonFile != null;
          addKey(project, jsonFile);
        }
        catch (IOException e) {
          throw new RuntimeException(e);
        }
      });
  }

  private void addKey(@NotNull Project project, JsonFile additionalJson) {
    if (!ReadonlyStatusHandler.ensureFilesWritable(project, additionalJson.getVirtualFile())) {
      return;
    }

    JsonElementGenerator generator = new JsonElementGenerator(project);
    JsonArray propertiesArray = findOrCreatePropertiesArray(generator, additionalJson);
    if (propertiesArray == null) {
      Messages.showWarningDialog(project,
                                 SpringBootApiBundle.message("spring.boot.properties.invalid.json",
                                                             additionalJson.getVirtualFile().getPath()),
                                 SpringBootApiBundle.message("spring.boot.name"));
      return;
    }

    WriteAction.run((ThrowableRunnable<IncorrectOperationException>)() -> {
      JsonObject value =
        generator.createValue("{\n" +
                              "  \"" + SpringBootMetadataConstants.NAME + "\": \"" + myKeyName + "\",\n" +
                              "  \"" + SpringBootMetadataConstants.TYPE + "\": \"java.lang.String\",\n" +
                              "  \"" + SpringBootMetadataConstants.DESCRIPTION + "\": \"Description for " + myKeyName + ".\"" +
                              "}");

      boolean hasValues = !propertiesArray.getValueList().isEmpty();
      if (hasValues) {
        propertiesArray.addBefore(generator.createComma(), propertiesArray.getLastChild());
      }

      JsonObject added = (JsonObject)propertiesArray.addBefore(value, propertiesArray.getLastChild());

      CodeStyleManager.getInstance(project)
        .reformatText(additionalJson, 0, additionalJson.getTextLength());

      added.navigate(true);
    });
  }

  @Nullable
  private static JsonArray findOrCreatePropertiesArray(JsonElementGenerator generator, JsonFile additionalJson) {
    JsonObject rootObject = ObjectUtils.tryCast(additionalJson.getTopLevelValue(), JsonObject.class);
    if (rootObject == null) return null;

    JsonProperty propertiesRoot = rootObject.findProperty(SpringBootMetadataConstants.PROPERTIES);
    if (propertiesRoot == null) {
      return WriteAction.compute((ThrowableComputable<JsonArray, IncorrectOperationException>)() -> {
        JsonProperty propertiesProperty =
          generator.createProperty(SpringBootMetadataConstants.PROPERTIES, "[]");
        if (!rootObject.getPropertyList().isEmpty()) {
          rootObject.addBefore(generator.createComma(), rootObject.getLastChild());
        }

        JsonProperty propertiesAdded = (JsonProperty)rootObject.addBefore(propertiesProperty, rootObject.getLastChild());
        return (JsonArray)propertiesAdded.getValue();
      });
    }

    return ObjectUtils.tryCast(propertiesRoot.getValue(), JsonArray.class);
  }
}
