// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.codeInsight.AutoPopupController;
import com.intellij.codeInsight.completion.*;
import com.intellij.codeInsight.lookup.*;
import com.intellij.codeInspection.ex.EditInspectionToolsSettingsAction;
import com.intellij.codeInspection.ex.InspectionProfileImpl;
import com.intellij.microservices.config.ConfigPlaceholderReference;
import com.intellij.openapi.actionSystem.IdeActions;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.keymap.KeymapUtil;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Couple;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.patterns.ElementPattern;
import com.intellij.profile.codeInspection.InspectionProfileManager;
import com.intellij.psi.PsiElement;
import com.intellij.spring.boot.SpringBootApiBundle;
import com.intellij.util.Consumer;
import com.intellij.util.PlatformIcons;
import com.intellij.util.ProcessingContext;
import icons.SpringBootApiIcons;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * Adds completion for configured replacement tokens and placeholder.
 * Additional registration via {@link LookupActionProvider} adds intention to open highlighting inspection's settings to configure them.
 */
public abstract class SpringBootReplacementTokenCompletionContributor extends CompletionContributor implements LookupActionProvider {

  @NotNull
  private final Key<Couple<String>> myTokenKey;

  @Nullable
  private final String myInspectionId;

  private final TokenInsertHandler myTokenInsertHandler = new TokenInsertHandler();

  /**
   * @param place Invocation place.
   * @return List valid {@code prefix|suffix} tokens.
   * @see SpringBootReplacementTokenStorage
   */
  @NotNull
  protected abstract List<Couple<String>> getReplacementTokens(@NotNull PsiElement place);

  /**
   * CTOR.
   *
   * @param place        Value pattern.
   * @param tokenKey     Unique key for storing data in lookup elements.
   * @param inspectionId (Optional) ID of inspection to open settings for configuring replacement tokens.
   */
  protected SpringBootReplacementTokenCompletionContributor(@NotNull ElementPattern<? extends PsiElement> place,
                                                            @NotNull Key<Couple<String>> tokenKey,
                                                            @Nullable String inspectionId) {
    myTokenKey = tokenKey;
    myInspectionId = inspectionId;

    extend(CompletionType.BASIC, place, new CompletionProvider<CompletionParameters>() {
      @Override
      protected void addCompletions(@NotNull CompletionParameters parameters,
                                    @NotNull ProcessingContext context,
                                    @NotNull CompletionResultSet result) {
        if (!parameters.isExtendedCompletion()) {
          final String shortcut = KeymapUtil.getFirstKeyboardShortcutText(IdeActions.ACTION_CODE_COMPLETION);
          result.addLookupAdvertisement(SpringBootApiBundle.message(
            "SpringBootReplacementTokenCompletionContributor.press.again.to.show.replacement.tokens", shortcut));
          return;
        }

        List<Couple<String>> replacementTokens = getReplacementTokens(parameters.getPosition());
        String text = parameters.getPosition().getText();
        for (Couple<String> token : replacementTokens) {
          if (StringUtil.contains(text, token.first)) {
            return;
          }
        }

        CompletionResultSet noPrefixResult = result.withPrefixMatcher("");
        for (Couple<String> token : replacementTokens) {
          addCompletionVariant(noPrefixResult, token, "Replacement token");
        }

        addCompletionVariant(noPrefixResult,
                             Couple.of(ConfigPlaceholderReference.PLACEHOLDER_PREFIX, ConfigPlaceholderReference.PLACEHOLDER_SUFFIX),
                             "Placeholder");
      }

      private void addCompletionVariant(CompletionResultSet completionResultSet,
                                        Couple<String> token,
                                        String typeText) {
        LookupElementBuilder lookupElement = LookupElementBuilder.create(token.first + token.second)
          .withPresentableText(token.first + "..." + token.second)
          .withBoldness(true)
          .withIcon(SpringBootApiIcons.SpringBoot)
          .withTypeText(typeText, true)
          .withInsertHandler(myTokenInsertHandler);
        lookupElement.putUserData(myTokenKey, token);
        completionResultSet.addElement(PrioritizedLookupElement.withPriority(lookupElement, -50));
      }
    });
  }

  @Override
  public void fillActions(LookupElement element, Lookup lookup, Consumer<LookupElementAction> consumer) {
    if (myInspectionId != null && myTokenKey.isIn(element)) {
      consumer.consume(new LookupElementAction(PlatformIcons.EDIT, SpringBootApiBundle.message(
        "SpringBootReplacementTokenCompletionContributor.configure.replacement.tokens")) {
        @Override
        public Result performLookupAction() {
          Project project = lookup.getProject();

          ApplicationManager.getApplication().invokeLater(() -> {
            InspectionProfileImpl profile = InspectionProfileManager.getInstance(project).getCurrentProfile();
            EditInspectionToolsSettingsAction.editToolSettings(project, profile, myInspectionId);
          }, project.getDisposed());

          return Result.HIDE_LOOKUP;
        }
      });
    }
  }


  private class TokenInsertHandler implements InsertHandler<LookupElement> {

    @Override
    public void handleInsert(@NotNull InsertionContext context, @NotNull LookupElement item) {
      Couple<String> token = item.getUserData(myTokenKey);
      assert token != null;

      context.getEditor().getCaretModel().moveCaretRelatively(-token.second.length(), 0, false, false, false);

      AutoPopupController.getInstance(context.getProject()).scheduleAutoPopup(context.getEditor());
    }
  }
}
