// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.codeInspection.ProblemHighlightType;
import com.intellij.codeInspection.ProblemsHolder;
import com.intellij.microservices.config.MetaConfigKey;
import com.intellij.microservices.config.MetaConfigKey.Deprecation;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.spring.boot.SpringBootApiBundle;
import com.intellij.spring.boot.application.metadata.additional.DefineLocalMetaConfigKeyFix;

public class SpringBootConfigFileHighlightingUtil {
  private final ProblemsHolder myHolder;

  public SpringBootConfigFileHighlightingUtil(ProblemsHolder holder) {
    myHolder = holder;
  }

  public void highlightDeprecatedConfigKey(PsiElement keyElement,
                                           MetaConfigKey configKey,
                                           LocalQuickFix... quickFixes) {
    final Deprecation deprecation = configKey.getDeprecation();
    final String reasonShortText = deprecation.getReason().getShortText();
    String reason = StringUtil.isNotEmpty(reasonShortText)
                    ? SpringBootApiBundle.message("SpringBootConfigFileHighlightingUtil.deprecated", reasonShortText)
                    : SpringBootApiBundle
                      .message("SpringBootConfigFileHighlightingUtil.deprecated.configuration.property", configKey.getName());

    ProblemHighlightType problemHighlightType = deprecation.getLevel() == Deprecation.DeprecationLevel.ERROR
                                                ? ProblemHighlightType.GENERIC_ERROR
                                                : ProblemHighlightType.LIKE_DEPRECATED;

    myHolder.registerProblem(keyElement,
                             reason,
                             problemHighlightType,
                             quickFixes);
  }

  public void highlightUnresolvedConfigKey(PsiElement keyElement, String qualifiedConfigKeyName) {
    myHolder.registerProblem(keyElement,
                             SpringBootApiBundle.message("SpringBootConfigFileHighlightingUtil.cannot.resolve.configuration.property", qualifiedConfigKeyName ),
                             ProblemHighlightType.GENERIC_ERROR_OR_WARNING,
                             new DefineLocalMetaConfigKeyFix(keyElement, qualifiedConfigKeyName));
  }
}
