/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.spi;

import com.intellij.lang.properties.IProperty;
import com.intellij.lang.properties.psi.impl.PropertiesFileImpl;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleServiceManager;
import com.intellij.psi.PsiClass;
import com.intellij.util.PairProcessor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * Provides access to {@code spring.factories} configuration files.
 *
 * @since 15
 */
public abstract class SpringSpiManager {

  public static SpringSpiManager getInstance(Module module) {
    return ModuleServiceManager.getService(module, SpringSpiManager.class);
  }

  /**
   * Returns all configuration files in current module.
   *
   * @param includeTests Include config files in test source roots.
   * @return List of files.
   */
  public abstract List<PropertiesFileImpl> getSpringFactoriesFiles(boolean includeTests);

  /**
   * Returns all resolved classes configured for given key.
   *
   * @param includeTests Include config files in test source roots.
   * @param key          Key to search for.
   * @return Resolved classes or empty list of none found.
   */
  public abstract List<PsiClass> getClassesListValue(boolean includeTests, @NotNull String key);

  /**
   * Process all resolved classes list values.
   *
   * @param includeTests Process config files in test source roots.
   * @param processor    Processor.
   * @param valueHint    Only process config files containing given text (e.g. current class FQN) or {@code null} to process all properties.
   * @return Processing result.
   * @since 2016.2
   */
  public abstract boolean processClassesListValues(boolean includeTests,
                                                   @Nullable String valueHint,
                                                   PairProcessor<IProperty, PsiClass> processor);
}
