/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.references;

import com.intellij.codeInsight.daemon.EmptyResolveMessageProvider;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.*;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.converters.SpringConverterUtil;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.utils.SpringModelUtils;
import com.intellij.util.ArrayUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Reference to Spring Bean by name, optionally limited to required base-class.
 */
public class SpringBeanReference extends PsiReferenceBase<PsiElement> implements EmptyResolveMessageProvider {

  @Nullable
  private final PsiClass myRequiredClass;
  private final boolean myFactoryBeanRef;

  public SpringBeanReference(final PsiElement element) {
    this(element, ElementManipulators.getManipulator(element).getRangeInElement(element));
  }

  public SpringBeanReference(final PsiElement element, final TextRange range) {
    this(element, range, null, false);
  }

  public SpringBeanReference(final PsiElement element,
                             TextRange range,
                             @Nullable final PsiClass requiredClass,
                             boolean isFactoryBeanRef) {
    super(element, range);
    myRequiredClass = requiredClass;
    myFactoryBeanRef = isFactoryBeanRef;
  }

  public boolean isFactoryBeanRef() {
    return myFactoryBeanRef;
  }

  @NotNull
  protected CommonSpringModel getSpringModel() {
    return SpringModelUtils.getInstance().getSpringModel(getElement());
  }

  public PsiElement resolve() {
    final String beanName = getValue();

    final CommonSpringModel springModel = getSpringModel();

    final SpringBeanPointer pointer = SpringModelSearchers.findBean(springModel, beanName);
    return pointer == null || !pointer.isValid() ? null : pointer.getPsiElement();
  }

  @Override
  public PsiElement bindToElement(@NotNull final PsiElement element) throws IncorrectOperationException {
    return getElement();
  }

  @NotNull
  public Object[] getVariants() {
    final CommonSpringModel model = getSpringModel();

    final Collection<SpringBeanPointer> beans;
    if (myRequiredClass != null && !CommonClassNames.JAVA_LANG_OBJECT.equals(myRequiredClass.getQualifiedName())) {
      final SpringModelSearchParameters.BeanClass searchParameters =
        SpringModelSearchParameters.byClass(myRequiredClass).withInheritors().effectiveBeanTypes();
      beans = SpringModelSearchers.findBeans(model, searchParameters);
    }
    else {
      beans = model.getAllCommonBeans();
    }

    List<LookupElement> lookups = new ArrayList<>(beans.size());
    for (SpringBeanPointer bean : beans) {
      ContainerUtil.addIfNotNull(lookups, SpringConverterUtil.createCompletionVariant(bean));
    }
    return ArrayUtil.toObjectArray(lookups);
  }

  @NotNull
  public String getUnresolvedMessagePattern() {
    return SpringApiBundle.message("model.bean.error.message", getValue());
  }
}
