/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.profiles;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReferenceBase;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

/**
 * @author Sergey Vasiliev
 * @see SpringProfilesFactory#getProfilesReferences(PsiElement, String, boolean)
 * @since 14.1
 */
public class SpringProfilePsiReference extends PsiReferenceBase<PsiElement> {
  private final String myProfileName;
  private final Module myModule;
  private final boolean myIsDefinition;

  public SpringProfilePsiReference(@NotNull PsiElement element,
                                   @NotNull TextRange range,
                                   @NotNull Module module,
                                   boolean definition) {
    super(element, range);
    myIsDefinition = definition;
    myProfileName = range.substring(element.getText());
    myModule = module;
  }

  public PsiElement resolve() {
    if (StringUtil.isEmptyOrSpaces(myProfileName)) return myElement;

    final SpringProfilesFactory profilesFactory = SpringProfilesFactory.getInstance(myModule);
    if (myIsDefinition) {
      return profilesFactory.getOrCreateProfile(myProfileName, getElement());
    }

    return profilesFactory.findProfile(myProfileName);
  }

  public boolean isSoft() {
    return true;
  }

  @NotNull
  public Object[] getVariants() {
    Set<SpringProfile> profiles = SpringProfilesFactory.getInstance(myModule).getProfiles();
    return profiles.toArray(new SpringProfile[profiles.size()]);
  }
}
