/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.xml;

import org.jetbrains.annotations.NotNull;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Determines {@link DomSpringBean} bean name.
 * <p/>
 * This can be done either:
 * <ol>
 * <li>statically via {@link #value()}</li>
 * <li>at runtime via {@link #provider()}</li>
 * </ol>
 * Setting both attributes is not allowed.
 *
 * @see DomSpringBeanImpl#getBeanName()
 * @since 14
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
public @interface BeanName {

  /**
   * Static bean name.
   *
   * @return Bean name.
   */
  @NotNull String value() default "";

  /**
   * Dynamic bean name.
   *
   * @return Provider class.
   */
  @NotNull Class<? extends BeanNameProvider> provider() default BeanNameProvider.class;

  /**
   * Whether the provided bean name is to be used for display purposes only.
   * <p/>
   * Usually used with "container" or "marker" beans having infrastructure role using "virtual name".
   *
   * @return {@code true} if name is not actual bean identifier.
   * @since 15
   */
  boolean displayOnly() default false;
}
