/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts.propertySources;

import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamClassMeta;
import com.intellij.jam.reflect.JamStringAttributeMeta;
import com.intellij.lang.properties.psi.PropertiesFile;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.jam.converters.PropertiesFileConverter;
import com.intellij.spring.model.jam.stereotype.SpringPropertySource;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;
import java.util.Set;

public class SpringTestPropertySource implements SpringPropertySource {

  protected static SemKey<SpringTestPropertySource>
    SPRING_TEST_PROPERTY_SOURCE_JAM_KEY = PROPERTY_SOURCE_JAM_KEY.subKey("SpringTestPropertySource");

  private static final JamStringAttributeMeta.Collection<Set<PropertiesFile>> VALUE_ATTR_META =
    JamAttributeMeta.collectionString("value", new PropertiesFileConverter());

  private static final JamStringAttributeMeta.Collection<Set<PropertiesFile>> LOCATIONS_ATTR_META =
    JamAttributeMeta.collectionString("locations", new PropertiesFileConverter());

  static final JamAnnotationMeta ANNO_META = new JamAnnotationMeta(SpringAnnotationsConstants.TEST_PROPERTY_SOURCE);

  public static final JamClassMeta<SpringTestPropertySource> META = new JamClassMeta<>(null, SpringTestPropertySource.class,
                                                                                       SPRING_TEST_PROPERTY_SOURCE_JAM_KEY);

  static {
    META.addAnnotation(ANNO_META);
    ANNO_META.addAttribute(VALUE_ATTR_META);
    ANNO_META.addAttribute(LOCATIONS_ATTR_META);
  }

  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final PsiClass myPsiClass;

  public SpringTestPropertySource(@NotNull PsiClass psiClass) {
    myPsiClass = psiClass;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiClass);
  }

  public SpringTestPropertySource(@NotNull PsiAnnotation annotation) {
    myPsiClass = PsiTreeUtil.getParentOfType(annotation, PsiClass.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  @NotNull
  @JamPsiConnector
  public PsiClass getPsiElement() {
    return myPsiClass;
  }

  public boolean isPsiValid() {
    return myPsiClass.isValid();
  }

  @Nullable
  public PsiAnnotation getAnnotation() {
    return ANNO_META.getAnnotation(getPsiElement());
  }

  public Set<PropertiesFile> getPropertiesFiles() {
    Set<PropertiesFile> propertiesFiles = ContainerUtil.newLinkedHashSet();
    if (getValueAttr().isEmpty() && getLocationsAttr().isEmpty()) {
      PropertiesFile propertiesFile = getDefaultPropertiesFile();
      if (propertiesFile != null) return Collections.singleton(propertiesFile);
    }
    else {
      addPropertiesFiles(propertiesFiles, getValueAttr());
      addPropertiesFiles(propertiesFiles, getLocationsAttr());
    }
    return propertiesFiles;
  }

  @NotNull
  protected List<JamStringAttributeElement<Set<PropertiesFile>>> getLocationsAttr() {
    return LOCATIONS_ATTR_META.getJam(myPsiAnnotation);
  }

  @NotNull
  protected List<JamStringAttributeElement<Set<PropertiesFile>>> getValueAttr() {
    return VALUE_ATTR_META.getJam(myPsiAnnotation);
  }

  protected void addPropertiesFiles(@NotNull Set<PropertiesFile> propertiesFiles,
                                    @NotNull List<JamStringAttributeElement<Set<PropertiesFile>>> attributeValues) {
    for (JamStringAttributeElement<Set<PropertiesFile>> attributeElement : attributeValues) {
      final Set<PropertiesFile> value = attributeElement.getValue();
      if (value != null) propertiesFiles.addAll(value);
    }
  }

  @Nullable
  private PropertiesFile getDefaultPropertiesFile() {
    final String propertiesFileName = getDefaultPropertiesFileName();
    final PsiDirectory containingDirectory = getPsiElement().getContainingFile().getContainingDirectory();
    if (containingDirectory != null) {
      PsiFile file = containingDirectory.findFile(propertiesFileName);
      if (file instanceof PropertiesFile) return (PropertiesFile)file;

      final PsiPackage psiPackage = JavaDirectoryService.getInstance().getPackage(containingDirectory);
      if (psiPackage != null) {
        for (PsiDirectory psiDirectory : psiPackage.getDirectories()) {
          file = psiDirectory.findFile(propertiesFileName);
          if (file instanceof PropertiesFile) return (PropertiesFile)file;
        }
      }
    }

    return null;
  }

  private String getDefaultPropertiesFileName() {
    return getPsiElement().getName() + ".properties";
  }
}
