/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.converters;

import com.intellij.jam.JamConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.ElementManipulators;
import com.intellij.psi.PsiLiteral;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.PsiReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PackageReferenceSet;
import com.intellij.spring.model.utils.SpringReferenceUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

public class PackageJamConverter extends JamConverter<Collection<PsiPackage>> {

  @Override
  public Collection<PsiPackage> fromString(@Nullable String s, JamStringAttributeElement<Collection<PsiPackage>> context) {
    if (StringUtil.isEmptyOrSpaces(s)) return null;

    final PsiLiteral psiLiteral = context.getPsiLiteral();
    if (psiLiteral == null) return null;

    return createReferenceSet(psiLiteral, s).resolvePackage();
  }

  @NotNull
  @Override
  public PsiReference[] createReferences(JamStringAttributeElement<Collection<PsiPackage>> context) {
    final PsiLiteral psiLiteral = context.getPsiLiteral();
    if (psiLiteral == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    final String value = context.getStringValue();
    if (value == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    return createReferenceSet(psiLiteral, value).getPsiReferences();
  }

  @NotNull
  private PackageReferenceSet createReferenceSet(@NotNull PsiLiteral psiLiteral, String value) {
    return new SpringAntPatternPackageReferenceSet(value, psiLiteral, ElementManipulators.getOffsetInElement(psiLiteral),
                                                   SpringReferenceUtils.getResolveScope(psiLiteral));
  }
}
