// Copyright 2000-2017 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.intellij.spring.model.highlighting.jam;

import com.intellij.codeInspection.*;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiJavaFile;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.model.utils.SpringModelUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

public abstract class AbstractSpringJavaConfigInspection extends AbstractBaseJavaLocalInspectionTool {

  @Nullable
  public ProblemDescriptor[] checkFile(@NotNull PsiFile file, @NotNull InspectionManager manager, boolean isOnTheFly) {
    if (JamCommonUtil.isPlainJavaFile(file)) {
      final Module module = ModuleUtilCore.findModuleForPsiElement(file);
      if (SpringCommonUtils.hasSpringFacet(module) || SpringModelUtils.getInstance().hasAutoConfiguredModels(module)) {

        final ProblemsHolder holder = new ProblemsHolder(manager, file, isOnTheFly);

        checkJavaFile((PsiJavaFile)file, holder, isOnTheFly, module);

        final List<ProblemDescriptor> problemDescriptors = holder.getResults();
        return problemDescriptors.toArray(new ProblemDescriptor[problemDescriptors.size()]);
      }
    }
    return null;
  }

  protected void checkJavaFile(@NotNull final PsiJavaFile javaFile,
                               @NotNull final ProblemsHolder holder,
                               final boolean isOnTheFly,
                               @NotNull Module module) {
    for (PsiClass psiClass : javaFile.getClasses()) {
      checkClassInternal(psiClass, holder, module);
    }
  }

  private void checkClassInternal(final PsiClass aClass, final ProblemsHolder holder, @NotNull Module module) {
    checkClass(aClass, holder, module);
    for (PsiClass psiClass : aClass.getInnerClasses()) {
      checkClass(psiClass, holder, module);
    }
  }

  protected abstract void checkClass(final PsiClass aClass, final ProblemsHolder holder, @NotNull Module module);
}
