/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.custom;

import com.intellij.openapi.extensions.Extensions;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleServiceManager;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.openapi.util.VolatileNotNullLazyValue;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider.Result;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.contexts.model.BeansSpringModel;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Set;

/**
 * @since 15
 */
public class CustomComponentsDiscovererHelper {
  @NotNull private final Module myModule;

  private final NotNullLazyValue<Set<CachedValue<CommonSpringModel>>> myCachedModels =
    new VolatileNotNullLazyValue<Set<CachedValue<CommonSpringModel>>>() {
      @NotNull
      protected Set<CachedValue<CommonSpringModel>> compute() {
        Set<CachedValue<CommonSpringModel>> set = ContainerUtil.newLinkedHashSet();

        for (CustomModuleComponentsDiscoverer discoverer : Extensions.getExtensions(CustomModuleComponentsDiscoverer.EP_NAME)) {
          if (discoverer.accepts(myModule)) {
            set.add(createCachedModel(discoverer));
          }
        }
        return set;
      }

      private CachedValue<CommonSpringModel> createCachedModel(@NotNull final CustomModuleComponentsDiscoverer discoverer) {
        return CachedValuesManager.getManager(myModule.getProject()).createCachedValue(() -> {
          final CommonSpringModel model =
            new CustomComponentDiscovererBeansModel(myModule, new NotNullLazyValue<Collection<? extends SpringBeanPointer>>() {
              @NotNull
              @Override
              protected Collection<? extends SpringBeanPointer> compute() {
                return BeanService.getInstance().mapSpringBeans(discoverer.getCustomComponents(myModule));
              }
            }, discoverer.getProviderName());
          return Result.create(model, discoverer.getDependencies(myModule));
        }, false);
      }
    };

  public static CustomComponentsDiscovererHelper getInstance(@NotNull Module module) {
    return ModuleServiceManager.getService(module, CustomComponentsDiscovererHelper.class);
  }

  public CustomComponentsDiscovererHelper(@NotNull Module module) {
    myModule = module;
  }

  public Collection<CommonSpringModel> getCustomModels() {
    return ContainerUtil.map(myCachedModels.getValue(), CachedValue::getValue);
  }


  private static class CustomComponentDiscovererBeansModel extends BeansSpringModel {

    @NotNull private final String myProviderName;

    private CustomComponentDiscovererBeansModel(@Nullable Module module,
                                                @NotNull NotNullLazyValue<Collection<? extends SpringBeanPointer>> pointers,
                                                @NotNull String providerName) {
      super(module, pointers);
      myProviderName = providerName;
    }

    @Override
    public String toString() {
      return myProviderName + ": " + super.toString();
    }
  }
}
