/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.DumbService;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.model.custom.CustomModuleComponentsDiscoverer;
import com.intellij.spring.model.utils.SpringCommonUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

/**
 * @author Yann C&eacute;bron
 */
public abstract class SpringImplicitBeansProviderBase extends CustomModuleComponentsDiscoverer {

  protected abstract Collection<CommonSpringBean> getImplicitBeans(@NotNull Module module);

  @NotNull
  @Override
  public final Collection<CommonSpringBean> getCustomComponents(@Nullable Module module) {
    if (module == null || module.isDisposed() || DumbService.isDumb(module.getProject())) {
      return Collections.emptyList();
    }
    return getImplicitBeans(module);
  }

  @Nullable
  protected PsiClass findClassInDependenciesAndLibraries(@NotNull Module module,
                                                                @NotNull String className) {
    if (module.isDisposed()) {
      return null;
    }

    final JavaPsiFacade facade = JavaPsiFacade.getInstance(module.getProject());
    return facade.findClass(className,
                            GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module, includeTests()));
  }

  protected boolean includeTests() {
    return false;
  }

  protected void addImplicitBean(@NotNull Collection<CommonSpringBean> implicitBeans,
                                 @NotNull Module module,
                                 @NotNull String className,
                                 @NotNull String beanName) {
    doAddImplicitBean(implicitBeans, module, className, beanName, false);
  }

  /**
   * Adds implicit bean with given name if <em>library</em> class exists.
   *
   * @param implicitBeans List of implicit beans.
   * @param module        Module.
   * @param className     FQN of library class.
   * @param beanName      Name of implicit bean.
   * @since 15
   */
  protected void addImplicitLibraryBean(@NotNull Collection<CommonSpringBean> implicitBeans,
                                        @NotNull Module module,
                                        @NotNull String className,
                                        @NotNull String beanName) {
    doAddImplicitBean(implicitBeans, module, className, beanName, true);
  }

  private void doAddImplicitBean(@NotNull Collection<CommonSpringBean> implicitBeans,
                                 @NotNull Module module,
                                 @NotNull String className,
                                 @NotNull String beanName,
                                 boolean isLibraryClass) {
    final PsiClass psiClass = isLibraryClass ? SpringCommonUtils.findLibraryClass(module, className) :
                              findClassInDependenciesAndLibraries(module, className);
    if (psiClass != null) {
      implicitBeans.add(new SpringImplicitBean(getProviderName(), psiClass, beanName));
    }
  }

  @Override
  public Object[] getDependencies(@NotNull Module module) {
    return SpringModificationTrackersManager.getInstance(module.getProject()).getOuterModelsDependencies();
  }
}
