/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.facet;

import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

class SpringFileSetData {

  private final String myId;
  private String myName;
  private final List<String> myFiles = new SmartList<>();
  private final Set<String> myDependencies = new LinkedHashSet<>();
  private Set<String> myActiveProfiles = new LinkedHashSet<>();
  private boolean myRemoved;

  static SpringFileSetData create(@NonNls @NotNull String id, @NotNull String name) {
    return new SpringFileSetData(id, name);
  }

  private SpringFileSetData(@NonNls @NotNull String id, @NotNull String name) {
    myId = id;
    myName = name;
  }

  public boolean isRemoved() {
    return myRemoved;
  }

  @NotNull
  public String getId() {
    return myId;
  }

  public String getName() {
    return myName;
  }

  public void setName(@NotNull final String name) {
    myName = name;
  }

  @NotNull
  public List<String> getFiles() {
    return myFiles;
  }

  public void setActiveProfiles(@NotNull Set<String> activeProfiles) {
    myActiveProfiles = activeProfiles;
  }

  @NotNull
  public Set<String> getActiveProfiles() {
    return myActiveProfiles;
  }

  public Set<String> getDependencies() {
    return myDependencies;
  }

  public void addDependency(final String dep) {
    myDependencies.add(dep);
  }

  public void addFile(@NonNls String url) {
    if (!StringUtil.isEmptyOrSpaces(url)) {
      myFiles.add(url);
    }
  }

  public void removeFile(@NonNls String url) {
    myFiles.remove(url);
  }

  void setRemoved(final boolean removed) {
    myRemoved = removed;
  }

  public static SpringFileSetData create(SpringFileSetData original) {
    SpringFileSetData springFileSetData = new SpringFileSetData(original.getId(), original.getName());
    springFileSetData.setRemoved(original.isRemoved());
    springFileSetData.getDependencies().addAll(original.getDependencies());
    springFileSetData.getFiles().addAll(original.getFiles());
    springFileSetData.getActiveProfiles().addAll(original.getActiveProfiles());

    return springFileSetData;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    SpringFileSetData data = (SpringFileSetData)o;

    if (!myId.equals(data.myId)) return false;
    if (myName != null ? !myName.equals(data.myName) : data.myName != null) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = myId.hashCode();
    result = 31 * result + (myName != null ? myName.hashCode() : 0);
    return result;
  }
}
