/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.AtomicNotNullLazyValue;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;

public class BeansSpringModel extends AbstractSimpleSpringModel {

  private final NotNullLazyValue<Collection<? extends SpringBeanPointer>> myPointers;
  private final Module myModule;

  public BeansSpringModel(@Nullable Module module, @NotNull final Set<PsiClass> classes) {
    this(module, new NotNullLazyValue<Collection<? extends SpringBeanPointer>>() {
      @NotNull
      @Override
      protected Collection<? extends SpringBeanPointer> compute() {
        final BeanService beanService = BeanService.getInstance();
        Set<SpringBeanPointer> pointers = new LinkedHashSet<>(classes.size());
        for (PsiClass psiClass : classes) {
          if (psiClass.isValid()) {
            pointers.add(beanService.createSpringBeanPointer(new CustomSpringComponent(psiClass)));
          }
        }
        return pointers;
      }
    });
  }

  public BeansSpringModel(@Nullable Module module, @NotNull NotNullLazyValue<Collection<? extends SpringBeanPointer>> pointers) {
    assert !(pointers instanceof AtomicNotNullLazyValue);
    myModule = module;
    myPointers = pointers;
  }

  @SuppressWarnings("unchecked")
  @Override
  protected Collection<SpringBeanPointer> calculateLocalBeans() {
    return (Collection<SpringBeanPointer>)myPointers.getValue();
  }

  @NotNull
  @Override
  public Collection<SpringBeanPointer> calculateDomBeans() {
    return Collections.emptySet();
  }

  @Nullable
  @Override
  public Module getModule() {
    return myModule;
  }

  @NotNull
  @Override
  public Set<String> getAllProfiles() {
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public Set<PsiFile> getConfigFiles() {
    return Collections.emptySet();
  }

  @Override
  public boolean hasConfigFile(@NotNull PsiFile configFile) {
    return false;
  }
}
