/*
 * Decompiled with CFR 0.152.
 */
package org.osgi.service.indexer.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import org.osgi.framework.Version;
import org.osgi.service.indexer.Builder;
import org.osgi.service.indexer.Capability;
import org.osgi.service.indexer.Requirement;
import org.osgi.service.indexer.Resource;
import org.osgi.service.indexer.ResourceAnalyzer;
import org.osgi.service.indexer.impl.EE;
import org.osgi.service.indexer.impl.GeneratorState;
import org.osgi.service.indexer.impl.MimeType;
import org.osgi.service.indexer.impl.URLResolver;
import org.osgi.service.indexer.impl.Util;
import org.osgi.service.indexer.impl.types.SymbolicName;
import org.osgi.service.indexer.impl.types.VersionKey;
import org.osgi.service.indexer.impl.types.VersionRange;
import org.osgi.service.indexer.impl.util.Hex;
import org.osgi.service.indexer.impl.util.OSGiHeader;
import org.osgi.service.indexer.impl.util.Yield;
import org.osgi.service.log.LogService;

public class BundleAnalyzer
implements ResourceAnalyzer {
    private static URI cwd = new File("").toURI().normalize();
    private static final String SHA_256 = "SHA-256";
    private static final String PROVIDE_CAPABILITY = "Provide-Capability";
    private static final String REQUIRE_CAPABILITY = "Require-Capability";
    private static final String IMPORT_SERVICE_AVAILABILITY = "availability:";
    private static final String SUFFIX_JAR = ".jar";
    private final ThreadLocal<GeneratorState> state = new ThreadLocal();
    private final LogService log;

    public BundleAnalyzer(LogService log) {
        this.log = log;
    }

    @Override
    public void analyzeResource(Resource resource, List<Capability> capabilities, List<Requirement> requirements) throws Exception {
        MimeType mimeType = Util.getMimeType(resource);
        if (mimeType == MimeType.Bundle || mimeType == MimeType.Fragment) {
            this.doBundleIdentity(resource, mimeType, capabilities);
            this.doContent(resource, mimeType, capabilities);
            this.doBundleAndHost(resource, capabilities);
            this.doExports(resource, capabilities);
            this.doImports(resource, requirements);
            this.doRequireBundles(resource, requirements);
            this.doFragment(resource, requirements);
            this.doExportService(resource, capabilities);
            this.doImportService(resource, requirements);
            this.doBREE(resource, requirements);
            this.doCapabilities(resource, capabilities);
            this.doRequirements(resource, requirements);
            this.doBundleNativeCode(resource, requirements);
        } else {
            this.doPlainJarIdentity(resource, capabilities);
            this.doContent(resource, mimeType, capabilities);
        }
    }

    private void doBundleIdentity(Resource resource, MimeType mimeType, List<? super Capability> caps) throws Exception {
        String type;
        Manifest manifest = resource.getManifest();
        if (manifest == null) {
            throw new IllegalArgumentException("Missing bundle manifest.");
        }
        switch (mimeType) {
            case Bundle: {
                type = "osgi.bundle";
                break;
            }
            case Fragment: {
                type = "osgi.fragment";
                break;
            }
            default: {
                type = "jarfile";
            }
        }
        SymbolicName bsn = Util.getSymbolicName(resource);
        boolean singleton = Boolean.TRUE.toString().equalsIgnoreCase(bsn.getAttributes().get("singleton:"));
        Version version = Util.getVersion(resource);
        Builder builder = new Builder().setNamespace("osgi.identity").addAttribute("osgi.identity", bsn.getName()).addAttribute("type", type).addAttribute("version", version);
        if (singleton) {
            builder.addDirective("singleton", Boolean.TRUE.toString());
        }
        caps.add(builder.buildCapability());
    }

    private void doPlainJarIdentity(Resource resource, List<? super Capability> caps) {
        String name = (String)resource.getProperties().get("name");
        if (name.toLowerCase().endsWith(SUFFIX_JAR)) {
            name = name.substring(0, name.length() - SUFFIX_JAR.length());
        }
        Version version = null;
        int dashIndex = name.lastIndexOf(45);
        if (dashIndex > 0) {
            try {
                String versionStr = name.substring(dashIndex + 1);
                version = new Version(versionStr);
                name = name.substring(0, dashIndex);
            }
            catch (Exception e) {
                version = null;
            }
        }
        Builder builder = new Builder().setNamespace("osgi.identity").addAttribute("osgi.identity", name).addAttribute("type", "jarfile");
        if (version != null) {
            builder.addAttribute("version", version);
        }
        caps.add(builder.buildCapability());
    }

    void setStateLocal(GeneratorState state) {
        this.state.set(state);
    }

    private GeneratorState getStateLocal() {
        return this.state.get();
    }

    private void doContent(Resource resource, MimeType mimeType, List<? super Capability> capabilities) throws Exception {
        String sha = this.calculateSHA(resource);
        List<String> locations = this.calculateLocation(resource);
        for (String location : locations) {
            Builder builder = new Builder().setNamespace("osgi.content");
            builder.addAttribute("osgi.content", sha);
            builder.addAttribute("url", location);
            long size = resource.getSize();
            if (size > 0L) {
                builder.addAttribute("size", size);
            }
            builder.addAttribute("mime", mimeType.toString());
            capabilities.add(builder.buildCapability());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String calculateSHA(Resource resource) throws IOException, NoSuchAlgorithmException {
        MessageDigest digest = MessageDigest.getInstance(SHA_256);
        byte[] buf = new byte[1024];
        try (InputStream stream = null;){
            int bytesRead;
            stream = resource.getStream();
            while ((bytesRead = stream.read(buf, 0, 1024)) >= 0) {
                digest.update(buf, 0, bytesRead);
            }
        }
        return Hex.toHexString(digest.digest());
    }

    private List<String> calculateLocation(Resource resource) throws IOException, URISyntaxException {
        URI absoluteDir;
        URI root;
        URI relativeDir;
        File file = new File(resource.getLocation()).getAbsoluteFile();
        URI normalizedUri = file.toURI().normalize();
        File normalizedFile = Paths.get(normalizedUri).toFile();
        GeneratorState state = this.getStateLocal();
        if (state == null) {
            return Collections.singletonList(cwd.relativize(normalizedUri).toString());
        }
        List<URLResolver> resolvers = state.getResolvers();
        if (resolvers != null && !resolvers.isEmpty()) {
            ArrayList<String> uris = new ArrayList<String>();
            for (URLResolver resolver : resolvers) {
                try {
                    URI uri = resolver.resolver(file);
                    if (uri != null) {
                        uris.add(uri.toString());
                        continue;
                    }
                    this.log.log(4, "Resolver " + resolver + " had no output for " + normalizedUri);
                }
                catch (Exception e) {
                    if (this.log == null) continue;
                    this.log.log(1, "Resolver " + resolver + " failed on " + normalizedUri);
                }
            }
            if (uris.isEmpty()) {
                this.log.log(3, "No URIs found by URI resolvers, falling back to old method");
            } else {
                return uris;
            }
        }
        if ((relativeDir = (root = state.getRootUrl()).relativize(absoluteDir = normalizedFile.getParentFile().toURI())) == absoluteDir) {
            throw new IllegalArgumentException("Cannot index files above the root URL. Root = " + root + " path is " + normalizedFile);
        }
        String fileName = normalizedFile.getName();
        String urlTemplate = state.getUrlTemplate();
        if (urlTemplate != null) {
            String bsn = urlTemplate.indexOf("%s") == -1 ? "" : Util.getSymbolicName(resource).getName();
            Version version = urlTemplate.indexOf("%v") == -1 ? Version.emptyVersion : Util.getVersion(resource);
            urlTemplate = urlTemplate.replaceAll("%s", "%1\\$s").replaceAll("%f", "%2\\$s").replaceAll("%p", "%3\\$s").replaceAll("%v", "%4\\$s");
            return Collections.singletonList(String.format(urlTemplate, bsn, fileName, relativeDir.toString(), version));
        }
        return Collections.singletonList(relativeDir.resolve(fileName).toString());
    }

    private void doBundleAndHost(Resource resource, List<? super Capability> caps) throws Exception {
        Builder bundleBuilder = new Builder().setNamespace("osgi.wiring.bundle");
        Builder hostBuilder = new Builder().setNamespace("osgi.wiring.host");
        boolean allowFragments = true;
        Attributes attribs = resource.getManifest().getMainAttributes();
        if (attribs.getValue("Fragment-Host") != null) {
            return;
        }
        SymbolicName bsn = Util.getSymbolicName(resource);
        Version version = Util.getVersion(resource);
        bundleBuilder.addAttribute("osgi.wiring.bundle", bsn.getName()).addAttribute("bundle-version", version);
        hostBuilder.addAttribute("osgi.wiring.host", bsn.getName()).addAttribute("bundle-version", version);
        for (Map.Entry<String, String> attribEntry : bsn.getAttributes().entrySet()) {
            String key = attribEntry.getKey();
            if (key.endsWith(":")) {
                String directiveName = key.substring(0, key.length() - 1);
                if ("fragment-attachment".equalsIgnoreCase(directiveName)) {
                    if (!"never".equalsIgnoreCase(attribEntry.getValue())) continue;
                    allowFragments = false;
                    continue;
                }
                if ("singleton".equalsIgnoreCase(directiveName)) continue;
                bundleBuilder.addDirective(directiveName, attribEntry.getValue());
                continue;
            }
            bundleBuilder.addAttribute(key, attribEntry.getValue());
        }
        caps.add(bundleBuilder.buildCapability());
        if (allowFragments) {
            caps.add(hostBuilder.buildCapability());
        }
    }

    private void doExports(Resource resource, List<? super Capability> caps) throws Exception {
        Manifest manifest = resource.getManifest();
        String exportsStr = manifest.getMainAttributes().getValue("Export-Package");
        Map<String, Map<String, String>> exports = OSGiHeader.parseHeader(exportsStr);
        for (Map.Entry<String, Map<String, String>> entry : exports.entrySet()) {
            Builder builder = new Builder().setNamespace("osgi.wiring.package");
            String pkgName = OSGiHeader.removeDuplicateMarker(entry.getKey());
            builder.addAttribute("osgi.wiring.package", pkgName);
            String versionStr = entry.getValue().get("version");
            Version version = versionStr != null ? new Version(versionStr) : new Version(0, 0, 0);
            builder.addAttribute("version", version);
            for (Map.Entry<String, String> attribEntry : entry.getValue().entrySet()) {
                String key = attribEntry.getKey();
                if ("specification-version".equalsIgnoreCase(key) || "version".equalsIgnoreCase(key)) continue;
                if (key.endsWith(":")) {
                    builder.addDirective(key.substring(0, key.length() - 1), attribEntry.getValue());
                    continue;
                }
                builder.addAttribute(key, attribEntry.getValue());
            }
            SymbolicName bsn = Util.getSymbolicName(resource);
            builder.addAttribute("bundle-symbolic-name", bsn.getName());
            Version bundleVersion = Util.getVersion(resource);
            builder.addAttribute("bundle-version", bundleVersion);
            caps.add(builder.buildCapability());
        }
    }

    private void doImports(Resource resource, List<? super Requirement> reqs) throws Exception {
        Manifest manifest = resource.getManifest();
        String importsStr = manifest.getMainAttributes().getValue("Import-Package");
        Map<String, Map<String, String>> imports = OSGiHeader.parseHeader(importsStr);
        for (Map.Entry<String, Map<String, String>> entry : imports.entrySet()) {
            StringBuilder filter = new StringBuilder();
            String pkgName = OSGiHeader.removeDuplicateMarker(entry.getKey());
            filter.append("(osgi.wiring.package=").append(pkgName).append(")");
            String versionStr = entry.getValue().get("version");
            if (versionStr != null) {
                VersionRange version = new VersionRange(versionStr);
                filter.insert(0, "(&");
                Util.addVersionFilter(filter, version, VersionKey.PackageVersion);
                filter.append(")");
            }
            Builder builder = new Builder().setNamespace("osgi.wiring.package").addDirective("filter", filter.toString());
            this.copyAttribsAndDirectives(entry.getValue(), builder, "version", "specification-version");
            reqs.add(builder.buildRequirement());
        }
    }

    private void copyAttribsAndDirectives(Map<String, String> input, Builder output, String ... ignores) {
        HashSet<String> ignoreSet = new HashSet<String>(Arrays.asList(ignores));
        for (Map.Entry<String, String> entry : input.entrySet()) {
            String key = entry.getKey();
            if (ignoreSet.contains(key)) continue;
            if (key.endsWith(":")) {
                String directive = key.substring(0, key.length() - 1);
                output.addDirective(directive, entry.getValue());
                continue;
            }
            output.addAttribute(key, entry.getValue());
        }
    }

    private void doRequireBundles(Resource resource, List<? super Requirement> reqs) throws Exception {
        Manifest manifest = resource.getManifest();
        String requiresStr = manifest.getMainAttributes().getValue("Require-Bundle");
        if (requiresStr == null) {
            return;
        }
        Map<String, Map<String, String>> requires = OSGiHeader.parseHeader(requiresStr);
        for (Map.Entry<String, Map<String, String>> entry : requires.entrySet()) {
            StringBuilder filter = new StringBuilder();
            String bsn = OSGiHeader.removeDuplicateMarker(entry.getKey());
            filter.append("(osgi.wiring.bundle=").append(bsn).append(")");
            String versionStr = entry.getValue().get("bundle-version");
            if (versionStr != null) {
                VersionRange version = new VersionRange(versionStr);
                filter.insert(0, "(&");
                Util.addVersionFilter(filter, version, VersionKey.BundleVersion);
                filter.append(")");
            }
            Builder builder = new Builder().setNamespace("osgi.wiring.bundle").addDirective("filter", filter.toString());
            this.copyAttribsAndDirectives(entry.getValue(), builder, "bundle-version");
            reqs.add(builder.buildRequirement());
        }
    }

    private void doFragment(Resource resource, List<? super Requirement> reqs) throws Exception {
        Manifest manifest = resource.getManifest();
        String fragmentHost = manifest.getMainAttributes().getValue("Fragment-Host");
        if (fragmentHost != null) {
            StringBuilder filter = new StringBuilder();
            Map<String, Map<String, String>> fragmentList = OSGiHeader.parseHeader(fragmentHost);
            if (fragmentList.size() != 1) {
                throw new IllegalArgumentException("Invalid Fragment-Host header: cannot contain multiple entries");
            }
            Map.Entry<String, Map<String, String>> entry = fragmentList.entrySet().iterator().next();
            String bsn = entry.getKey();
            filter.append("(&(osgi.wiring.host=").append(bsn).append(")");
            String versionStr = entry.getValue().get("bundle-version");
            VersionRange version = versionStr != null ? new VersionRange(versionStr) : new VersionRange(Version.emptyVersion.toString());
            Util.addVersionFilter(filter, version, VersionKey.BundleVersion);
            filter.append(")");
            Builder builder = new Builder().setNamespace("osgi.wiring.host").addDirective("filter", filter.toString());
            reqs.add(builder.buildRequirement());
        }
    }

    private void doExportService(Resource resource, List<? super Capability> caps) throws Exception {
        String exportsStr = resource.getManifest().getMainAttributes().getValue("Export-Service");
        Map<String, Map<String, String>> exports = OSGiHeader.parseHeader(exportsStr);
        for (Map.Entry<String, Map<String, String>> export : exports.entrySet()) {
            String service = OSGiHeader.removeDuplicateMarker(export.getKey());
            Builder builder = new Builder().setNamespace("osgi.service").addAttribute("objectClass", service);
            for (Map.Entry<String, String> attribEntry : export.getValue().entrySet()) {
                builder.addAttribute(attribEntry.getKey(), attribEntry.getValue());
            }
            builder.addDirective("effective", "active");
            caps.add(builder.buildCapability());
        }
    }

    private void doImportService(Resource resource, List<? super Requirement> reqs) throws Exception {
        String importsStr = resource.getManifest().getMainAttributes().getValue("Import-Service");
        Map<String, Map<String, String>> imports = OSGiHeader.parseHeader(importsStr);
        for (Map.Entry<String, Map<String, String>> imp : imports.entrySet()) {
            String service = OSGiHeader.removeDuplicateMarker(imp.getKey());
            Map<String, String> attribs = imp.getValue();
            boolean optional = false;
            String availabilityStr = attribs.get(IMPORT_SERVICE_AVAILABILITY);
            if ("optional".equals(availabilityStr)) {
                optional = true;
            }
            StringBuilder filter = new StringBuilder();
            filter.append('(').append("objectClass").append('=').append(service).append(')');
            Builder builder = new Builder().setNamespace("osgi.service").addDirective("filter", filter.toString()).addDirective("effective", "active");
            if (optional) {
                builder.addDirective("resolution", "optional");
            }
            reqs.add(builder.buildRequirement());
        }
    }

    private void doBREE(Resource resource, List<? super Requirement> reqs) throws Exception {
        String breeStr = resource.getManifest().getMainAttributes().getValue("Bundle-RequiredExecutionEnvironment");
        Map<String, Map<String, String>> brees = OSGiHeader.parseHeader(breeStr);
        if (!brees.isEmpty()) {
            String filter;
            if (brees.size() == 1) {
                String bree = brees.keySet().iterator().next();
                filter = EE.parseBREE(bree).toFilter();
            } else {
                StringBuilder builder = new StringBuilder().append("(|");
                for (String bree : brees.keySet()) {
                    bree = OSGiHeader.removeDuplicateMarker(bree);
                    builder.append(EE.parseBREE(bree).toFilter());
                }
                builder.append(')');
                filter = builder.toString();
            }
            Requirement requirement = new Builder().setNamespace("osgi.ee").addDirective("filter", filter).buildRequirement();
            reqs.add(requirement);
        }
    }

    private void doCapabilities(Resource resource, final List<? super Capability> caps) throws Exception {
        String capsStr = resource.getManifest().getMainAttributes().getValue(PROVIDE_CAPABILITY);
        BundleAnalyzer.buildFromHeader(capsStr, new Yield<Builder>(){

            @Override
            public void yield(Builder builder) {
                caps.add(builder.buildCapability());
            }
        });
    }

    private void doRequirements(Resource resource, final List<? super Requirement> reqs) throws IOException {
        String reqsStr = resource.getManifest().getMainAttributes().getValue(REQUIRE_CAPABILITY);
        BundleAnalyzer.buildFromHeader(reqsStr, new Yield<Builder>(){

            @Override
            public void yield(Builder builder) {
                reqs.add(builder.buildRequirement());
            }
        });
    }

    private void doBundleNativeCode(Resource resource, List<? super Requirement> reqs) throws IOException {
        Object builder;
        String filter;
        String nativeHeaderStr = resource.getManifest().getMainAttributes().getValue("Bundle-NativeCode");
        if (nativeHeaderStr == null) {
            return;
        }
        boolean optional = false;
        LinkedList<String> options = new LinkedList<String>();
        Map<String, Map<String, String>> nativeHeader = OSGiHeader.parseHeader(nativeHeaderStr);
        for (Map.Entry<String, Map<String, String>> entry : nativeHeader.entrySet()) {
            String selectionFilter;
            String languageFilter;
            String processorFilter;
            String versionRangeStr;
            String name = entry.getKey();
            if ("*".equals(name)) {
                optional = true;
                continue;
            }
            StringBuilder builder2 = new StringBuilder().append("(&");
            Map<String, String> attribs = entry.getValue();
            String osnamesFilter = this.buildFilter(attribs, "osname", "osgi.native.osname");
            if (osnamesFilter != null) {
                builder2.append(osnamesFilter);
            }
            if ((versionRangeStr = attribs.get("osversion")) != null) {
                Util.addVersionFilter(builder2, new VersionRange(versionRangeStr), VersionKey.NativeOsVersion);
            }
            if ((processorFilter = this.buildFilter(attribs, "processor", "osgi.native.processor")) != null) {
                builder2.append(processorFilter);
            }
            if ((languageFilter = this.buildFilter(attribs, "language", "osgi.native.language")) != null) {
                builder2.append(languageFilter);
            }
            if ((selectionFilter = attribs.get("selection-filter")) != null) {
                builder2.append(selectionFilter);
            }
            builder2.append(")");
            options.add(builder2.toString());
        }
        if (options.isEmpty()) {
            return;
        }
        if (options.size() == 1) {
            filter = (String)options.get(0);
        } else {
            builder = new StringBuilder();
            ((StringBuilder)builder).append("(|");
            for (String option : options) {
                ((StringBuilder)builder).append(option);
            }
            ((StringBuilder)builder).append(")");
            filter = ((StringBuilder)builder).toString();
        }
        builder = new Builder().setNamespace("osgi.native").addDirective("filter", filter);
        if (optional) {
            ((Builder)builder).addDirective("resolution", "optional");
        }
        reqs.add(((Builder)builder).buildRequirement());
    }

    private String buildFilter(Map<String, String> attribs, String match, String filterKey) {
        LinkedList<String> options = new LinkedList<String>();
        for (Map.Entry<String, String> entry : attribs.entrySet()) {
            String key = OSGiHeader.removeDuplicateMarker(entry.getKey());
            if (!match.equals(key)) continue;
            String filter = String.format("(%s~=%s)", filterKey, entry.getValue());
            options.add(filter);
        }
        if (options.isEmpty()) {
            return null;
        }
        if (options.size() == 1) {
            return (String)options.get(0);
        }
        StringBuilder builder = new StringBuilder();
        builder.append("(|");
        for (String option : options) {
            builder.append(option);
        }
        builder.append(")");
        return builder.toString();
    }

    private static void buildFromHeader(String headerStr, Yield<Builder> output) {
        if (headerStr == null) {
            return;
        }
        Map<String, Map<String, String>> header = OSGiHeader.parseHeader(headerStr);
        for (Map.Entry<String, Map<String, String>> entry : header.entrySet()) {
            String namespace = OSGiHeader.removeDuplicateMarker(entry.getKey());
            Builder builder = new Builder().setNamespace(namespace);
            Map<String, String> attribs = entry.getValue();
            Util.copyAttribsToBuilder(builder, attribs);
            output.yield(builder);
        }
    }
}

