/*
 * Decompiled with CFR 0.152.
 */
package org.jetbrains.dekaf.jdbc.pooling;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import javax.sql.DataSource;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.dekaf.core.ImplementationAccessibleService;
import org.jetbrains.dekaf.jdbc.pooling.ConnectionPoolExhaustedException;
import org.jetbrains.dekaf.jdbc.pooling.ConnectionPoolIsNotReadyException;
import org.jetbrains.dekaf.jdbc.pooling.ConnectionPoolOperationInterruptedException;
import org.jetbrains.dekaf.jdbc.pooling.UnexpectedDataSourceException;
import org.jetbrains.dekaf.util.Objects;

public class ConnectionPool
implements ImplementationAccessibleService {
    @NotNull
    private final DataSource myOriginalDataSource;
    private final CopyOnWriteArrayList<Connection> myAllConnections = new CopyOnWriteArrayList();
    private final LinkedBlockingDeque<Connection> myFreeConnections = new LinkedBlockingDeque();
    private volatile int myHoldConnections = 1;
    private volatile int myConnectionsLimit = 10;
    private int myBorrowTimeBeforeReplenish = 70;
    private int myBorrowTimeOut = 60000;
    private volatile boolean myReady;

    public ConnectionPool(@NotNull DataSource originalDataSource) {
        this.myOriginalDataSource = originalDataSource;
    }

    @NotNull
    public DataSource getOriginalDataSource() {
        return this.myOriginalDataSource;
    }

    public void connect() throws SQLException {
        int n = this.myHoldConnections - this.myAllConnections.size();
        if (this.myAllConnections.isEmpty()) {
            n = Math.max(n, 1);
        }
        for (int i = 0; i < n; ++i) {
            this.obtainOneConnectionIntoPool();
        }
        this.myReady = !this.myAllConnections.isEmpty();
    }

    private synchronized void obtainOneConnectionIntoPool() throws SQLException {
        if (this.myAllConnections.size() >= this.myConnectionsLimit) {
            return;
        }
        Connection connection = this.myOriginalDataSource.getConnection();
        if (connection == null) {
            throw new UnexpectedDataSourceException("DataSource " + this.myOriginalDataSource.getClass().getName() + " returned null.", "<DataSource.getConnection()>");
        }
        try {
            this.prepareConnectionAfterConnected(connection);
        }
        catch (SQLException sqle) {
            ConnectionPool.closeConnection(connection);
            throw sqle;
        }
        this.myAllConnections.add(connection);
        if (this.myReady) {
            ConnectionPool.sleep(11);
        }
        this.myFreeConnections.offer(connection);
    }

    protected void prepareConnectionAfterConnected(@NotNull Connection connection) throws SQLException {
        connection.setAutoCommit(true);
    }

    public boolean isReady() {
        return this.myReady;
    }

    @NotNull
    public Connection borrow() throws SQLException {
        if (!this.myReady) {
            throw new ConnectionPoolIsNotReadyException("The connection pool is not connected yet or may be is disconnecting.");
        }
        Connection connection = this.provideWithConnection();
        this.activateConnection(connection);
        return connection;
    }

    @NotNull
    private Connection provideWithConnection() throws SQLException {
        try {
            Connection connection = this.myFreeConnections.poll(this.myBorrowTimeBeforeReplenish, TimeUnit.MILLISECONDS);
            if (connection != null) {
                return connection;
            }
            while (this.myAllConnections.size() < this.myConnectionsLimit) {
                this.obtainOneConnectionIntoPool();
                connection = this.myFreeConnections.poll(this.myBorrowTimeBeforeReplenish, TimeUnit.MILLISECONDS);
                if (connection == null) continue;
                return connection;
            }
            connection = this.myFreeConnections.poll(this.myBorrowTimeOut, TimeUnit.MILLISECONDS);
            if (connection != null) {
                return connection;
            }
            int n = this.myAllConnections.size();
            throw new ConnectionPoolExhaustedException("The Connection Pool exhausted: all " + n + " connections are borrowed and not returned yet.");
        }
        catch (InterruptedException ie) {
            throw new ConnectionPoolOperationInterruptedException("Operation interrupted", (Throwable)ie, "<provide with connection>");
        }
    }

    protected void activateConnection(@NotNull Connection connection) throws SQLException {
    }

    public void release(@NotNull Connection connection) {
        try {
            this.passivateConnection(connection);
            this.myFreeConnections.offerFirst(connection);
        }
        catch (Exception e) {
            ConnectionPool.panic("Passivate Connection", e);
            this.closeAndLeaveConnection(connection);
        }
    }

    protected void passivateConnection(@NotNull Connection connection) throws SQLException {
        if (!connection.getAutoCommit()) {
            connection.rollback();
            connection.setAutoCommit(true);
        }
    }

    public void disconnect() {
        this.myReady = false;
        while (!this.myAllConnections.isEmpty()) {
            ConnectionPool.sleep(11);
            ArrayList toClose = new ArrayList(this.myConnectionsLimit);
            this.myFreeConnections.drainTo(toClose, 100);
            this.myAllConnections.removeAll(toClose);
            for (Connection connection : toClose) {
                ConnectionPool.closeConnection(connection);
            }
            toClose.clear();
        }
    }

    private void closeAndLeaveConnection(@NotNull Connection connection) {
        this.myFreeConnections.remove(connection);
        this.myAllConnections.remove(connection);
        ConnectionPool.closeConnection(connection);
        this.myReady = this.myReady && !this.myAllConnections.isEmpty();
    }

    private static void closeConnection(Connection connection) {
        try {
            connection.close();
        }
        catch (Exception e) {
            ConnectionPool.panic("Close connection", e);
        }
    }

    @Override
    @Nullable
    public <I> I getSpecificService(@NotNull Class<I> serviceClass, @NotNull String serviceName) throws ClassCastException {
        if (serviceName.toUpperCase().startsWith("JDBC")) {
            if (this.myOriginalDataSource instanceof ImplementationAccessibleService) {
                return ((ImplementationAccessibleService)((Object)this.myOriginalDataSource)).getSpecificService(serviceClass, serviceName);
            }
            return null;
        }
        if (serviceName.equalsIgnoreCase("connection-pool")) {
            return Objects.castTo(serviceClass, this);
        }
        return null;
    }

    public int getHoldConnections() {
        return this.myHoldConnections;
    }

    public void setHoldConnections(int holdConnections) {
        this.myHoldConnections = holdConnections;
    }

    public int getConnectionsLimit() {
        return this.myConnectionsLimit;
    }

    public void setConnectionsLimit(int connectionsLimit) {
        this.myConnectionsLimit = connectionsLimit;
    }

    public int countAllConnections() {
        return this.myAllConnections.size();
    }

    public int countFreeConnections() {
        return this.myFreeConnections.size();
    }

    public int countBorrowedConnections() {
        int z;
        int a = this.myAllConnections.size();
        int f = this.myFreeConnections.size();
        do {
            z = a - f;
        } while (a != this.myAllConnections.size() || f != this.myFreeConnections.size());
        return z;
    }

    private static void sleep(int milliseconds) {
        try {
            Thread.sleep(milliseconds);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private static void panic(String operationDescription, Exception e) {
        System.err.printf("Operation %s failed: %s: %s\n", operationDescription, e.getClass().getSimpleName(), e.getMessage());
    }
}

